<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

if (!function_exists('json_ok')) {
    function json_ok($data, array $meta = []) {
        echo json_encode(['data' => $data, 'meta' => array_merge(['ok' => true], $meta)], JSON_UNESCAPED_UNICODE);
        exit;
    }
}
if (!function_exists('json_err')) {
    function json_err(string $message, array $meta = []) {
        echo json_encode(['data' => null, 'meta' => array_merge(['ok' => false, 'error' => $message], $meta)], JSON_UNESCAPED_UNICODE);
        exit;
    }
}
if (!function_exists('get_param')) {
    function get_param(string $k, $default = null) {
        return $_GET[$k] ?? $_POST[$k] ?? $default;
    }
}

/**
 * 1) پیدا کردن و require کردن bootstrap/db/config های پروژه
 */
$debug = [
    'loaded_files' => [],
    'tried_files'  => [],
    'conn_source'  => null,
];

$dirs = [];
$cur = __DIR__;
for ($i = 0; $i < 7; $i++) {
    $dirs[] = $cur;
    $parent = dirname($cur);
    if ($parent === $cur) break;
    $cur = $parent;
}

$candidateFiles = [
    // خیلی رایج‌ها
    '_bootstrap.php', 'bootstrap.php', '_init.php', 'init.php', 'common.php',
    // db helpers
    '_db.php', 'db.php', 'database.php', 'Db.php',
    // config
    'config.php', 'config.local.php', 'local.config.php',
    // پوشه‌های رایج
    'inc/bootstrap.php', 'includes/bootstrap.php', 'lib/bootstrap.php',
    'inc/db.php', 'includes/db.php', 'lib/db.php',
    'inc/config.php', 'includes/config.php', 'lib/config.php',
    // گاهی تو api هست
    'api/_bootstrap.php', 'api/bootstrap.php', 'api/_db.php', 'api/db.php', 'api/config.php',
];

$loadedReturnConfigs = []; // اگر config.php array return کنه

foreach ($dirs as $d) {
    foreach ($candidateFiles as $rel) {
        $path = $d . DIRECTORY_SEPARATOR . $rel;
        $debug['tried_files'][] = $path;
        if (is_file($path)) {
            // require_once ممکنه return array بده (config)
            $ret = require_once $path;
            $debug['loaded_files'][] = $path;

            if (is_array($ret)) {
                $loadedReturnConfigs[] = ['file' => $path, 'config' => $ret];
            }
        }
    }
}

/**
 * 2) اگر پروژه role/auth داره و توابعش لود شده، استفاده کن
 */
if (function_exists('require_role')) {
    require_role('admin');
}

/**
 * 3) گرفتن کانکشن از پروژه (اگر قبلاً جایی ساخته شده)
 */
$conn = null;

// حالت‌های رایج متغیر سراسری
$globalsToCheck = ['conn', 'db_conn', 'db', 'sqlsrv', 'pdo'];
foreach ($globalsToCheck as $g) {
    if (isset($GLOBALS[$g]) && $GLOBALS[$g]) {
        $conn = $GLOBALS[$g];
        $debug['conn_source'] = '$GLOBALS[' . $g . ']';
        break;
    }
}

// حالت‌های رایج توابع
if (!$conn) {
    $fnCandidates = ['db', 'get_db', 'db_conn', 'get_conn', 'sqlsrv_conn', 'connect_db'];
    foreach ($fnCandidates as $fn) {
        if (function_exists($fn)) {
            try {
                $tmp = $fn();
                if ($tmp) {
                    $conn = $tmp;
                    $debug['conn_source'] = $fn . '()';
                    break;
                }
            } catch (\Throwable $e) {
                // ignore
            }
        }
    }
}

/**
 * 4) اگر هنوز conn نداریم، از config array ها بسازیم
 * پشتیبانی از چند مدل کلید (db/server/uid/pwd و ...)
 */
if (!$conn) {
    $bestCfg = null;

    // اول config هایی که return array داشتند
    foreach ($loadedReturnConfigs as $item) {
        $cfg = $item['config'];

        // حالت: ['db'=>[...]]
        if (isset($cfg['db']) && is_array($cfg['db'])) {
            $bestCfg = $cfg['db'];
            $debug['conn_source'] = 'config_return:' . $item['file'] . ' (key=db)';
            break;
        }

        // حالت: مستقیم
        if (isset($cfg['server']) || isset($cfg['host'])) {
            $bestCfg = $cfg;
            $debug['conn_source'] = 'config_return:' . $item['file'] . ' (direct)';
            break;
        }
    }

    if ($bestCfg) {
        $server = $bestCfg['server'] ?? $bestCfg['host'] ?? 'localhost';
        $dbName = $bestCfg['database'] ?? $bestCfg['db'] ?? $bestCfg['dbname'] ?? null;
        $uid    = $bestCfg['uid'] ?? $bestCfg['user'] ?? $bestCfg['username'] ?? null;
        $pwd    = $bestCfg['pwd'] ?? $bestCfg['pass'] ?? $bestCfg['password'] ?? null;

        if ($dbName && $uid !== null && $pwd !== null) {
            $conn = sqlsrv_connect($server, [
                'Database' => $dbName,
                'UID'      => $uid,
                'PWD'      => $pwd,
                'CharacterSet' => 'UTF-8',
                'ReturnDatesAsStrings' => true,
                'MultipleActiveResultSets' => true,
            ]);

            if ($conn) {
                $debug['conn_source'] = $debug['conn_source'] ?: 'sqlsrv_connect(config)';
            }
        }
    }
}

/**
 * 4) اگر هنوز conn نداریم، fallback اتصال مستقیم (مثل پروژه‌های قبلی خودت)
 * توجه: DB اصلی شما طبق SSMS: DarianDW_Dashboard
 */
if (!$conn) {
    // اگر خواستی بعداً از بیرون ست کنی:
    // setx DASHBOARD_DB_NAME DarianDW_Dashboard
    // setx DASHBOARD_DB_UID SH
    // setx DASHBOARD_DB_PWD 1!2@3#4$5%6^
    // setx DASHBOARD_DB_SERVER localhost

    $tryServers = array_filter([
        getenv('DASHBOARD_DB_SERVER') ?: null,
        'localhost',
        '127.0.0.1',
    ]);

    $tryDbs = array_filter([
        getenv('DASHBOARD_DB_NAME') ?: null,
        'DarianDW_Dashboard',
        'DarianDW', // اگه بعضی جدول‌ها اونجاست
    ]);

    $uid = getenv('DASHBOARD_DB_UID') ?: 'SH';
    $pwd = getenv('DASHBOARD_DB_PWD') ?: '1!2@3#4$5%6^';

    $optionsBase = [
        'CharacterSet'             => 'UTF-8',
        'ReturnDatesAsStrings'     => true,
        'MultipleActiveResultSets' => true,
    ];

    foreach ($tryServers as $svr) {
        foreach ($tryDbs as $dbName) {
            $conn = sqlsrv_connect($svr, $optionsBase + [
                'Database' => $dbName,
                'UID'      => $uid,
                'PWD'      => $pwd,
            ]);

            if ($conn) {
                $debug['conn_source'] = "fallback_sqlsrv_connect({$svr} / {$dbName})";
                break 2;
            }
        }
    }
}

if (!$conn) {
    json_err('DB connection not available. (conn is null)', [
        'debug' => [
            'loaded_files' => $debug['loaded_files'],
            'conn_source'  => $debug['conn_source'],
        ],
        'sqlsrv_errors' => sqlsrv_errors(),
    ]);
}


/**
 * 5) پارامترها (date_to اینجا inclusive فرض می‌کنیم)
 */
$date_from = (string)get_param('date_from', '');
$date_to   = (string)get_param('date_to', '');
$l1_id     = get_param('l1_id', null);
$store_id  = get_param('store_id', null);

if ($date_from === '' || $date_to === '') {
    json_err('date_from and date_to are required');
}

// rule: همه آنلاین بجز 4 = حضوری
$IN_PERSON_ID = 4;

/**
 * 6) Query از جدول تجمیعی
 * DateKey نوع DATE هست → date_to inclusive
 */
$sql = "
SELECT
    SUM(CASE WHEN a.SaleTypeID = ? THEN CAST(a.SalesAmount AS DECIMAL(38,0)) ELSE CAST(0 AS DECIMAL(38,0)) END) AS inperson_sales,
    SUM(CASE WHEN a.SaleTypeID <> ? OR a.SaleTypeID IS NULL THEN CAST(a.SalesAmount AS DECIMAL(38,0)) ELSE CAST(0 AS DECIMAL(38,0)) END) AS online_sales
FROM BI.AggDailyStoreBrandL1 a
WHERE a.DateKey >= ? AND a.DateKey <= ?
  AND (? IS NULL OR a.L1ID = ?)
  AND (? IS NULL OR a.StoreID = ?)
";

$params = [
    $IN_PERSON_ID,
    $IN_PERSON_ID,
    $date_from,
    $date_to,
    $l1_id, $l1_id,
    $store_id, $store_id,
];

$stmt = sqlsrv_query($conn, $sql, $params);
if (!$stmt) {
    json_err('DB error (online_vs_inperson)', [
        'sqlsrv_errors' => sqlsrv_errors(),
        'debug' => $debug,
    ]);
}

$row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC) ?: [];

$online   = (string)($row['online_sales'] ?? "0");
$inperson = (string)($row['inperson_sales'] ?? "0");

// total
$totalDec = (float)$online + (float)$inperson;
$onlinePct   = $totalDec > 0 ? round(((float)$online / $totalDec) * 100, 1) : 0;
$inpersonPct = $totalDec > 0 ? round(((float)$inperson / $totalDec) * 100, 1) : 0;

json_ok([
    'total_sales' => (string)((int)$totalDec), // برای سادگی خروجی
    'items' => [
        [
            'channel' => 'online',
            'label' => 'آنلاین',
            'sales' => $online,
            'share_percent' => $onlinePct,
        ],
        [
            'channel' => 'in_person',
            'label' => 'حضوری',
            'sales' => $inperson,
            'share_percent' => $inpersonPct,
        ],
    ],
], [
    'filters' => [
        'date_from' => $date_from,
        'date_to' => $date_to,
        'l1_id' => $l1_id,
        'store_id' => $store_id,
    ],
    'source' => 'BI.AggDailyStoreBrandL1',
    'amount_unit' => 'IRR',
    'debug' => [
        'conn_source' => $debug['conn_source'],
        'loaded_files_count' => count($debug['loaded_files']),
    ]
]);
