<?php
// /dashboard-project/api/products/top6m.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$__t0 = microtime(true);

// ------------------------- helpers -------------------------
function out(array $payload, int $status = 200): void {
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}
function q(?string $k, $default = null) {
    return isset($_GET[$k]) && $_GET[$k] !== '' ? $_GET[$k] : $default;
}
function is_ymd(string $s): bool {
    return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $s);
}
function intv($v, int $min, int $max, int $def): int {
    if ($v === null || $v === '') return $def;
    if (!preg_match('/^-?\d+$/', (string)$v)) return $def;
    $n = (int)$v;
    if ($n < $min) return $min;
    if ($n > $max) return $max;
    return $n;
}
function sql_one($conn, string $sql, array $params = []): ?array {
    $st = sqlsrv_query($conn, $sql, $params);
    if (!$st) return null;
    $row = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC);
    sqlsrv_free_stmt($st);
    return $row ?: null;
}
function col_exists($conn, string $fullTable, string $col): bool {
    $sql = "SELECT CASE WHEN COL_LENGTH(?, ?) IS NULL THEN 0 ELSE 1 END AS ok";
    $row = sql_one($conn, $sql, [$fullTable, $col]);
    return (bool)($row['ok'] ?? 0);
}
function pick_first_col($conn, string $fullTable, array $candidates): ?string {
    foreach ($candidates as $c) {
        if (col_exists($conn, $fullTable, $c)) return $c;
    }
    return null;
}
function table_exists($conn, string $fullTable): bool {
    $sql = "SELECT CASE WHEN OBJECT_ID(?) IS NULL THEN 0 ELSE 1 END AS ok";
    $row = sql_one($conn, $sql, [$fullTable]);
    return (bool)($row['ok'] ?? 0);
}
function connect_sqlsrv(): array {
    // تلاش برای خواندن کانفیگ‌های محتمل پروژه
    $try = [
        __DIR__ . '/../config.php',
        __DIR__ . '/../../config.php',
        __DIR__ . '/../_config.php',
        __DIR__ . '/../../config.php',
    ];

    $cfg = null;
    foreach ($try as $p) {
        if (is_file($p)) {
            $tmp = include $p; // باید array برگردونه
            if (is_array($tmp) && isset($tmp['db'])) { $cfg = $tmp; break; }
            // بعضی پروژه‌ها مستقیم db رو برمی‌گردونن
            if (is_array($tmp) && (isset($tmp['server']) || isset($tmp['database']))) {
                $cfg = ['db' => $tmp];
                break;
            }
        }
    }

    // fallback (اگر کانفیگ پیدا نشد، اینا رو با دیتابیس خودت هماهنگ کن)
    $db = $cfg['db'] ?? [
        'server'   => 'localhost',
        'database' => 'DarianDW_Dashboard',
        'uid'      => 'SH',
        'pwd'      => '1!2@3#4$5%6^',
        'options'  => [
            'CharacterSet' => 'UTF-8',
            'ReturnDatesAsStrings' => true,
        ],
    ];

    $server   = $db['server']   ?? 'localhost';
    $database = $db['database'] ?? 'DarianDW_Dashboard';
    $uid      = $db['uid']      ?? null;
    $pwd      = $db['pwd']      ?? null;

    $options = $db['options'] ?? [];
    // اطمینان از ReturnDatesAsStrings برای JSON
    if (!isset($options['ReturnDatesAsStrings'])) $options['ReturnDatesAsStrings'] = true;
    if (!isset($options['CharacterSet'])) $options['CharacterSet'] = 'UTF-8';

    $connInfo = array_merge(
        ['Database' => $database],
        $uid !== null ? ['UID' => $uid] : [],
        $pwd !== null ? ['PWD' => $pwd] : [],
        $options
    );

    $conn = sqlsrv_connect($server, $connInfo);

    return [$conn, $server, $database];
}

// ------------------------- input -------------------------
$date_from = (string)q('date_from', '');
$date_to   = (string)q('date_to', '');
$l1_id     = q('l1_id', null);
$limit     = intv(q('limit', 6), 1, 50, 6);

if (!is_ymd($date_from) || !is_ymd($date_to)) {
    out([
        'data' => null,
        'meta' => ['ok' => false, 'error' => 'Invalid date_from/date_to. Expected YYYY-MM-DD.']
    ], 400);
}

$l1_id_int = null;
if ($l1_id !== null) {
    $l1_id_int = intv($l1_id, 1, 12, 0);
    if ($l1_id_int === 0) $l1_id_int = null;
}

// L1 MAP (ثابت و امن؛ دیگه join لازم نیست)
$l1_map = [
    1 => 'نوشیدنی',
    2 => 'لبنیات',
    3 => 'شوینده و بهداشتی',
    4 => 'کمپوت و کنسرو',
    5 => 'یخچالی و انجمادی',
    6 => 'چاشنی و افزودنی ها',
    7 => 'دخانیات',
    8 => 'خشکبار و تنقلات',
    9 => 'دسر و شیرینی',
    10 => 'چای ، قهوه ، محصولات پودری',
    11 => 'لوازم آشپزخانه ، مصرفی و فرهنگی',
    12 => 'خواروبار و کالای اساسی',
];

// ------------------------- connect -------------------------
[$conn, $server, $dbName] = connect_sqlsrv();

if (!$conn) {
    out([
        'data' => null,
        'meta' => [
            'ok' => false,
            'error' => 'DB connection not available. (conn is null)',
            'sqlsrv_errors' => sqlsrv_errors(),
            'debug' => ['server' => $server, 'database' => $dbName],
        ],
    ], 500);
}

// ------------------------- detect columns/tables -------------------------
$saleTable = 'BI.SaleDetail';

// ستون‌های محتمل نام محصول داخل SaleDetail (از همین استفاده می‌کنیم که join نخواد)
$productNameCol = pick_first_col($conn, $saleTable, [
    'ProductName', 'ProductTitle', 'KalaName', 'ItemName', 'GoodName'
]);

// برای نام برند: سعی می‌کنیم از dimها پیدا کنیم
$brandTable = null;
$brandIdCol = null;
$brandNameCol = null;

$candidateBrandTables = [
    'BI.DimBrand',
    'BI.Brand',
    'dbo.Brand',
    'dbo.brands',
];

foreach ($candidateBrandTables as $t) {
    if (!table_exists($conn, $t)) continue;

    $idCol = pick_first_col($conn, $t, ['BrandID', 'brand_id', 'Id', 'ID']);
    $nmCol = pick_first_col($conn, $t, ['BrandName', 'BrandTitle', 'Title', 'Name', 'brand_name']);

    if ($idCol && $nmCol) {
        $brandTable = $t;
        $brandIdCol = $idCol;
        $brandNameCol = $nmCol;
        break;
    }
}

// ------------------------- build SQL -------------------------
$filters = [];
$params  = [];

$filters[] = "sd.FactorDate >= ? AND sd.FactorDate < ?";
$params[] = $date_from;
$params[] = $date_to;

if ($l1_id_int !== null) {
    $filters[] = "sd.ProductGroupID = ?";
    $params[]  = $l1_id_int;
}

$where = implode(' AND ', $filters);

// اگر ProductNameCol نداریم، فقط NULL برمی‌گردونیم
$selectProductName = $productNameCol ? "MAX(sd.{$productNameCol})" : "CAST(NULL AS NVARCHAR(200))";

// join برند اگر پیدا شد
$joinBrand = "";
$selectBrandName = "CAST(NULL AS NVARCHAR(200)) AS brand_name";
if ($brandTable && $brandIdCol && $brandNameCol) {
    $joinBrand = "LEFT JOIN {$brandTable} b ON b.{$brandIdCol} = x.brand_id";
    $selectBrandName = "MAX(b.{$brandNameCol}) AS brand_name";
}

$sql = "
;WITH x AS (
    SELECT
        sd.ProductID       AS product_id,
        sd.BrandID         AS brand_id,
        sd.ProductGroupID  AS l1_id,
        {$selectProductName} AS product_name,
        SUM(CAST(sd.TotalPrice AS DECIMAL(38,0)))     AS sales,
        SUM(CAST(sd.Quantity   AS DECIMAL(18,3)))     AS qty
    FROM {$saleTable} sd
    WHERE {$where}
      AND sd.ProductID IS NOT NULL
      AND sd.BrandID IS NOT NULL
      AND sd.ProductGroupID IS NOT NULL
    GROUP BY sd.ProductID, sd.BrandID, sd.ProductGroupID
)
SELECT TOP {$limit}
    x.product_id,
    x.product_name,
    x.brand_id,
    {$selectBrandName},
    x.l1_id,
    x.sales,
    x.qty,
    CAST(100.0 * x.sales / NULLIF(SUM(x.sales) OVER(), 0) AS DECIMAL(6,2)) AS share_percent,
    SUM(x.sales) OVER() AS total_sales
FROM x
{$joinBrand}
GROUP BY
    x.product_id, x.product_name, x.brand_id, x.l1_id, x.sales, x.qty
ORDER BY x.sales DESC
OPTION (RECOMPILE);
";

$st = sqlsrv_query($conn, $sql, $params);
if (!$st) {
    out([
        'data' => null,
        'meta' => [
            'ok' => false,
            'error' => 'Query failed',
            'sqlsrv_errors' => sqlsrv_errors(),
            'debug' => [
                'database' => $dbName,
                'productNameCol' => $productNameCol,
                'brandTable' => $brandTable,
                'brandIdCol' => $brandIdCol,
                'brandNameCol' => $brandNameCol,
            ],
        ],
    ], 500);
}

$items = [];
$total_sales = 0;

while ($r = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC)) {
    $total_sales = (int)($r['total_sales'] ?? 0);

    $l1 = (int)($r['l1_id'] ?? 0);
    $items[] = [
        'product_id'    => (int)$r['product_id'],
        'product_name'  => $r['product_name'] !== null ? (string)$r['product_name'] : null,
        'brand_id'      => (int)$r['brand_id'],
        'brand_name'    => $r['brand_name'] !== null ? (string)$r['brand_name'] : null,
        'l1_id'         => $l1,
        'l1_title'      => $l1_map[$l1] ?? null,
        'sales'         => (string)$r['sales'], // DECIMAL بزرگه، رشته می‌دیم که JS خراب نکنه
        'qty'           => (string)$r['qty'],
        'share_percent' => isset($r['share_percent']) ? (float)$r['share_percent'] : 0.0,
    ];
}

sqlsrv_free_stmt($st);
sqlsrv_close($conn);

$out = [
    'data' => [
        'total_sales' => (string)$total_sales,
        'items' => $items,
    ],
    'meta' => [
        'ok' => true,
        'filters' => [
            'date_from' => $date_from,
            'date_to'   => $date_to,
            'l1_id'     => $l1_id_int,
            'limit'     => $limit,
        ],
        'source' => 'BI.SaleDetail (aggregated) + (optional Brand dim) + PHP L1_MAP',
        'duration_ms' => (int)round((microtime(true) - $__t0) * 1000),
    ],
];

out($out);
