<?php
// dashboard-project/api/products/top6m.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

// --- bootstrap (طبق پروژه خودت تنظیم کن)
$root = dirname(__DIR__); // .../api
$boot = $root . '/_bootstrap.php';
if (file_exists($boot)) require_once $boot;

// fallback minimal helpers اگر پروژه‌ات اینارو داشت حذفش کن
function jexit($arr){ echo json_encode($arr, JSON_UNESCAPED_UNICODE); exit; }
function qget($k,$def=null){ return isset($_GET[$k]) && $_GET[$k]!=='' ? $_GET[$k] : $def; }

// اگر تو پروژه‌ات db() داری همونو نگه دار
if (!function_exists('db')) {
    function db(){
        // اینجا باید به کانکشن پروژه‌ات وصل بشه (تو خودت داری)
        return null;
    }
}

$conn = db();
if (!$conn) {
    jexit(["data"=>null,"meta"=>["ok"=>false,"error"=>"DB connection not available. (conn is null)"]]);
}

// -------- inputs
$date_from = qget('date_from');
$date_to   = qget('date_to');
$limit     = (int)qget('limit', 6);
$l1_id     = qget('l1_id');      // optional
$sale_type = qget('sale_type_id'); // optional

if (!$date_from || !$date_to) {
    jexit(["data"=>null,"meta"=>["ok"=>false,"error"=>"date_from و date_to اجباری است"]]);
}
if ($limit <= 0 || $limit > 50) $limit = 6;

// -------- detect columns in BI.SaleDetail for titles
function table_has_col($conn, string $schema, string $table, string $col): bool {
    $sql = "SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA=? AND TABLE_NAME=? AND COLUMN_NAME=?";
    $stmt = sqlsrv_query($conn, $sql, [$schema,$table,$col]);
    if (!$stmt) return false;
    $row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);
    return (bool)$row;
}
function pick_col($conn, array $candidates): ?string {
    foreach ($candidates as $c) {
        if (table_has_col($conn,'BI','SaleDetail',$c)) return $c;
    }
    return null;
}

$productTitleCol = pick_col($conn, ['ProductTitle','ProductName','ItemName','GoodsName','KalaName','Product']);
$brandTitleCol   = pick_col($conn, ['BrandTitle','BrandName','Brand']);
$l1TitleCol      = pick_col($conn, ['ProductGroupTitle','ProductGroupName','L1Title']);

// اگر اسم‌ها تو SaleDetail نبود، فقط id برمی‌گردونیم (ولی کرش نمی‌کنه)
$selectProductTitle = $productTitleCol ? "MAX(sd.$productTitleCol) AS product_title" : "CAST(NULL AS nvarchar(255)) AS product_title";
$selectBrandTitle   = $brandTitleCol   ? "MAX(sd.$brandTitleCol)   AS brand_title"   : "CAST(NULL AS nvarchar(255)) AS brand_title";
$selectL1Title      = $l1TitleCol      ? "MAX(sd.$l1TitleCol)      AS l1_title"      : "CAST(NULL AS nvarchar(255)) AS l1_title";

// -------- query
// 1) top products از AggDailyProduct در بازه
$filters = [];
$whereAgg = "ap.DateKey >= ? AND ap.DateKey < ?";
$filters[] = $date_from;
$filters[] = $date_to;

if ($l1_id !== null) {
    $whereAgg .= " AND ap.L1ID = ?";
    $filters[] = (int)$l1_id;
}
if ($sale_type !== null) {
    $whereAgg .= " AND ap.SaleTypeID = ?";
    $filters[] = (int)$sale_type;
}

// Top list
$sqlTop = "
WITH topP AS (
    SELECT TOP ($limit)
        ap.ProductID,
        MAX(ap.BrandID) AS BrandID,
        MAX(ap.L1ID)    AS L1ID,
        SUM(ap.SalesAmount) AS total_sales
    FROM BI.AggDailyProduct ap
    WHERE $whereAgg
    GROUP BY ap.ProductID
    ORDER BY SUM(ap.SalesAmount) DESC
)
SELECT * FROM topP
ORDER BY total_sales DESC;
";

$stmtTop = sqlsrv_query($conn, $sqlTop, $filters);
if (!$stmtTop) {
    jexit(["data"=>null,"meta"=>["ok"=>false,"error"=>"DB error (top products)","sqlsrv_errors"=>sqlsrv_errors()]]);
}
$tops = [];
while ($r = sqlsrv_fetch_array($stmtTop, SQLSRV_FETCH_ASSOC)) $tops[] = $r;

if (!$tops) {
    jexit([
        "data"=>["items"=>[], "months"=>[], "series"=>[]],
        "meta"=>["ok"=>true,"filters"=>["date_from"=>$date_from,"date_to"=>$date_to,"limit"=>$limit,"l1_id"=>$l1_id,"sale_type_id"=>$sale_type], "source"=>"BI.AggDailyProduct"]
    ]);
}

$productIds = array_map(fn($x)=> (int)$x['ProductID'], $tops);

// 2) monthly series برای همون productIds
$in = implode(',', array_fill(0, count($productIds), '?'));
$paramsSeries = array_merge([$date_from,$date_to], $productIds);
if ($l1_id !== null) $paramsSeries[] = (int)$l1_id;
if ($sale_type !== null) $paramsSeries[] = (int)$sale_type;

$where2 = "ap.DateKey >= ? AND ap.DateKey < ? AND ap.ProductID IN ($in)";
if ($l1_id !== null)   $where2 .= " AND ap.L1ID = ?";
if ($sale_type !== null) $where2 .= " AND ap.SaleTypeID = ?";

$sqlSeries = "
SELECT
    CONVERT(char(7), ap.DateKey, 120) AS ym,
    ap.ProductID,
    SUM(ap.SalesAmount) AS sales
FROM BI.AggDailyProduct ap
WHERE $where2
GROUP BY CONVERT(char(7), ap.DateKey, 120), ap.ProductID
ORDER BY ym, ap.ProductID;
";

$stmtSer = sqlsrv_query($conn, $sqlSeries, $paramsSeries);
if (!$stmtSer) {
    jexit(["data"=>null,"meta"=>["ok"=>false,"error"=>"DB error (series)","sqlsrv_errors"=>sqlsrv_errors()]]);
}

$series = [];
$monthsSet = [];
while ($r = sqlsrv_fetch_array($stmtSer, SQLSRV_FETCH_ASSOC)) {
    $ym = $r['ym'];
    $monthsSet[$ym] = true;
    $pid = (int)$r['ProductID'];
    $series[$pid][$ym] = (string)$r['sales'];
}
$months = array_keys($monthsSet);
sort($months);

// 3) title lookup از SaleDetail (فقط برای همین productIds)
$paramsNames = $productIds;
$sqlNames = "
SELECT
    sd.ProductID,
    MAX(sd.BrandID) AS BrandID,
    MAX(sd.ProductGroupID) AS L1ID,
    $selectProductTitle,
    $selectBrandTitle,
    $selectL1Title
FROM BI.SaleDetail sd
WHERE sd.ProductID IN ($in)
GROUP BY sd.ProductID;
";

$stmtNames = sqlsrv_query($conn, $sqlNames, $paramsNames);
if (!$stmtNames) {
    jexit(["data"=>null,"meta"=>["ok"=>false,"error"=>"DB error (names)","sqlsrv_errors"=>sqlsrv_errors()]]);
}

$names = [];
while ($r = sqlsrv_fetch_array($stmtNames, SQLSRV_FETCH_ASSOC)) {
    $names[(int)$r['ProductID']] = $r;
}

// ---- output items (top order)
$items = [];
foreach ($tops as $t) {
    $pid = (int)$t['ProductID'];
    $nm  = $names[$pid] ?? [];
    $items[] = [
        "product_id"    => $pid,
        "product_title" => $nm['product_title'] ?? null,
        "brand_id"      => isset($t['BrandID']) ? (int)$t['BrandID'] : null,
        "brand_title"   => $nm['brand_title'] ?? null,
        "l1_id"         => isset($t['L1ID']) ? (int)$t['L1ID'] : null,
        "l1_title"      => $nm['l1_title'] ?? null,
        "total_sales"   => (string)$t['total_sales'],
    ];
}

// series normalized (fill missing months with 0)
$outSeries = [];
foreach ($productIds as $pid) {
    $row = [];
    foreach ($months as $m) $row[] = (string)($series[$pid][$m] ?? "0");
    $outSeries[] = ["product_id"=>$pid, "values"=>$row];
}

jexit([
    "data" => [
        "months" => $months,
        "items"  => $items,
        "series" => $outSeries
    ],
    "meta" => [
        "ok" => true,
        "filters" => [
            "date_from"=>$date_from,
            "date_to"=>$date_to,
            "limit"=>$limit,
            "l1_id"=>$l1_id,
            "sale_type_id"=>$sale_type
        ],
        "source" => "BI.AggDailyProduct + (names from BI.SaleDetail)"
    ]
]);
