<?php
// /api/products/performance_monthly.php
// Page2 Chart5: monthly performance of top N products (+ L1), default N=6

if (!defined('BI_API')) { define('BI_API', true); }

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

$t0 = microtime(true);

try {
    $user  = auth_user();                  // در dev => admin
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();                 // از همون db.php پروژه
} catch (Throwable $e) {
    fail('DB connection failed', 500, ['exception' => $e->getMessage()]);
}

// --- L1 ثابت (12 تا) ---
$L1_MAP = [
  1  => 'نوشیدنی',
  2  => 'لبنیات',
  3  => 'شوینده و بهداشتی',
  4  => 'کمپوت و کنسرو',
  5  => 'یخچالی و انجمادی',
  6  => 'چاشنی و افزودنی ها',
  7  => 'دخانیات',
  8  => 'خشکبار و تنقلات',
  9  => 'دسر و شیرینی',
  10 => 'چای ، قهوه ، محصولات پودری',
  11 => 'لوازم آشپزخانه ، مصرفی و فرهنگی',
  12 => 'خواروبار و کالای اساسی',
];

// inputs
$date_from = parse_date(get_param('date_from'));
$date_to   = parse_date(get_param('date_to')); // inclusive

// دیفالت: 6 ماه اخیر
if (!$date_from || !$date_to) {
    $date_to   = date('Y-m-d');
    $date_from = date('Y-m-01', strtotime('-5 months'));
}
$date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

$limit = (int)get_param('limit', 6);
if ($limit <= 0) $limit = 6;
if ($limit > 20) $limit = 20;

$l1_id        = get_param('l1_id');
$brand_id     = get_param('brand_id');
$province     = trim((string)get_param('province', ''));
$city         = trim((string)get_param('city', ''));
$area         = trim((string)get_param('area', ''));
$store_id     = get_param('store_id');
$sale_type_id = get_param('sale_type_id');

// scope enforcement
if (!empty($scope['province'])) $province = $scope['province'];
if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

$l1_id        = ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null;
$brand_id     = ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null;
$store_id     = ($store_id !== null && $store_id !== '') ? (int)$store_id : null;
$sale_type_id = ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null;

// Join DimStore only if we need geo filters
$joinStore = "";
$whereStore = "";
$paramsStore = [];

if ($province !== '' || $city !== '' || $area !== '') {
    $joinStore = "JOIN BI.DimStore ds WITH (NOLOCK) ON ds.StoreID = sd.StoreID";
    if ($province !== '') { $whereStore .= " AND LTRIM(RTRIM(ds.Province)) = LTRIM(RTRIM(?)) "; $paramsStore[] = $province; }
    if ($city !== '')     { $whereStore .= " AND LTRIM(RTRIM(ds.City)) = LTRIM(RTRIM(?)) ";     $paramsStore[] = $city; }
    if ($area !== '')     { $whereStore .= " AND CAST(ds.Area AS NVARCHAR(50)) = CAST(? AS NVARCHAR(50)) "; $paramsStore[] = $area; }
}

$where = "WHERE sd.FactorDate >= CAST(? AS DATETIME2(0)) AND sd.FactorDate < CAST(? AS DATETIME2(0))";
$params = [$date_from . " 00:00:00", $date_to_excl . " 00:00:00"];

if ($brand_id !== null)     { $where .= " AND sd.BrandID = ?";        $params[] = $brand_id; }
if ($l1_id !== null)        { $where .= " AND sd.ProductGroupID = ?"; $params[] = $l1_id; }
if ($store_id !== null)     { $where .= " AND sd.StoreID = ?";        $params[] = $store_id; }
if ($sale_type_id !== null) { $where .= " AND sd.SaleTypeID = ?";     $params[] = $sale_type_id; }

$where .= $whereStore;
$params = array_merge($params, $paramsStore);

// 1) Top N products (by sales)
$sqlTop = "
SELECT TOP ($limit)
    sd.ProductID AS product_id,
    MAX(sd.ProductTitle) AS product_title,
    MAX(sd.ProductGroupID) AS l1_id,
    SUM(TRY_CONVERT(DECIMAL(38,0), sd.TotalPrice)) AS total_sales
FROM BI.SaleDetail sd WITH (NOLOCK)
$joinStore
$where
GROUP BY sd.ProductID
ORDER BY SUM(TRY_CONVERT(DECIMAL(38,0), sd.TotalPrice)) DESC
OPTION (RECOMPILE)
";

$stmt = sqlsrv_query($conn, $sqlTop, $params);
if ($stmt === false) fail('DB error (products top)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

$top = [];
$productIds = [];
while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
    $pid = (int)$r['product_id'];
    $productIds[] = $pid;
    $l1 = (int)($r['l1_id'] ?? 0);
    $top[$pid] = [
        'product_id'    => $pid,
        'product_title' => (string)($r['product_title'] ?? ''),
        'l1_id'         => $l1,
        'l1_title'      => $L1_MAP[$l1] ?? null,
        'total_sales'   => (string)($r['total_sales'] ?? '0'),
        'series'        => [],
    ];
}
sqlsrv_free_stmt($stmt);

if (!$productIds) {
    json_response([
        'data' => ['items' => []],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'filters' => [
                'date_from' => $date_from, 'date_to' => $date_to, 'limit' => $limit,
                'l1_id' => $l1_id, 'brand_id' => $brand_id,
                'province' => $province ?: null, 'city' => $city ?: null, 'area' => $area ?: null,
                'store_id' => $store_id, 'sale_type_id' => $sale_type_id
            ],
            'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
            'amount_unit' => 'IRR',
        ]
    ]);
}

// 2) Monthly series for those products
$in = implode(',', array_fill(0, count($productIds), '?'));
$sqlSeries = "
SELECT
    sd.ProductID AS product_id,
    ym = CONVERT(CHAR(7), sd.FactorDate, 120),
    SUM(TRY_CONVERT(DECIMAL(38,0), sd.TotalPrice)) AS sales
FROM BI.SaleDetail sd WITH (NOLOCK)
$joinStore
$where
AND sd.ProductID IN ($in)
GROUP BY sd.ProductID, CONVERT(CHAR(7), sd.FactorDate, 120)
ORDER BY ym
OPTION (RECOMPILE)
";

$paramsSeries = array_merge($params, $productIds);

$stmt2 = sqlsrv_query($conn, $sqlSeries, $paramsSeries);
if ($stmt2 === false) fail('DB error (products monthly series)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

while ($r = sqlsrv_fetch_array($stmt2, SQLSRV_FETCH_ASSOC)) {
    $pid = (int)$r['product_id'];
    if (!isset($top[$pid])) continue;
    $top[$pid]['series'][] = [
        'ym'    => (string)$r['ym'],
        'sales' => (string)($r['sales'] ?? '0'),
    ];
}
sqlsrv_free_stmt($stmt2);

json_response([
    'data' => [
        'items' => array_values($top),
    ],
    'meta' => [
        'ok' => true,
        'role' => $user['Role'] ?? null,
        'filters' => [
            'date_from' => $date_from, 'date_to' => $date_to, 'limit' => $limit,
            'l1_id' => $l1_id, 'brand_id' => $brand_id,
            'province' => $province ?: null, 'city' => $city ?: null, 'area' => $area ?: null,
            'store_id' => $store_id, 'sale_type_id' => $sale_type_id
        ],
        'source' => 'BI.SaleDetail',
        'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
        'amount_unit' => 'IRR',
    ]
]);
