<?php
// /api/brands/top3m.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}
function date_add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // ---------- dates (default: last 90 days ending at max DateKey) ----------
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if (!$date_to) {
        $stmtMax = sqlsrv_query($conn, "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1");
        if ($stmtMax === false) sqlsrv_fail('DB error (max date)', 500);
        $mx = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC)['mx'] ?? null;
        if (!$mx) fail("No data to determine date range", 400);
        $date_to = (string)$mx;
    }
    if (!$date_from) {
        // 90 روز: شامل امروز => 89 روز قبل
        $date_from = date_add_days($date_to, -89);
    }
    $date_to_excl = date_add_days($date_to, 1);

    // ---------- filters ----------
    $limit = (int)get_param('limit', 6);
    if ($limit <= 0 || $limit > 50) $limit = 6;

    $l1_id = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // city/area filters (صفحه ۲: شهر/منطقه)
    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // search brand title (اختیاری)
    $q = trim((string)get_param('q', ''));

    // role scope
    if (!empty($scope['province'])) $province = $scope['province'];
    $brand_scope = null;
    if (!empty($scope['brand_id'])) $brand_scope = (int)$scope['brand_id'];

    // ---------- build WHERE + JOIN DimStore/DimBrand only if needed ----------
    $joinStore = "";
    $joinBrand = "LEFT JOIN BI.DimBrand db ON db.BrandID = a.BrandID";

    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($l1_id !== null && $l1_id !== '') { $where .= " AND a.L1ID = ?"; $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // اگر فیلتر مکانی داریم، از DimStore استفاده کن
    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
    }

    // اگر مدیر برند باشه، فقط برند خودش
    if ($brand_scope !== null) {
        $where .= " AND a.BrandID = ?";
        $params[] = $brand_scope;
    }

    // اگر سرچ برند داریم (q)، روی DimBrand اعمال کن
    if ($q !== '') {
        $where .= " AND (db.BrandTitle LIKE ?)";
        $params[] = '%' . $q . '%';
    }

    // ---------- query: top brands + total sales for share% ----------
    $sql = "
        WITH totals AS (
            SELECT
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $joinBrand
            $where
        ),
        topb AS (
            SELECT TOP ($limit)
                a.BrandID AS brand_id,
                ISNULL(NULLIF(LTRIM(RTRIM(db.BrandTitle)),''), CONCAT(N'Brand #', a.BrandID)) AS brand_title,
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $joinBrand
            $where
            GROUP BY a.BrandID, db.BrandTitle
            ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC
        )
        SELECT
            t.brand_id,
            t.brand_title,
            CAST(t.sales AS varchar(50)) AS sales,
            CAST(
                (CAST(t.sales AS decimal(38,6)) * 100) / NULLIF(CAST(tt.total_sales AS decimal(38,6)), 0)
            AS decimal(9,2)) AS share_percent,
            CAST(tt.total_sales AS varchar(50)) AS total_sales
        FROM topb t
        CROSS JOIN totals tt
        ORDER BY t.sales DESC
        OPTION (RECOMPILE);
    ";

    // چون $where دو بار تکرار شده (totals و topb)، params هم دو بار باید ارسال شود
    $params2 = array_merge($params, $params);

    $stmt = sqlsrv_query($conn, $sql, $params2);
    if ($stmt === false) sqlsrv_fail("DB error (top3m brands)", 500);

    $items = [];
    $total_sales = "0";
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $total_sales = (string)($r['total_sales'] ?? "0");
        $items[] = [
            'brand_id' => (int)$r['brand_id'],
            'brand_title' => (string)$r['brand_title'],
            'sales' => (string)($r['sales'] ?? "0"),
            'share_percent' => (float)($r['share_percent'] ?? 0),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'total_sales' => $total_sales,
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'range' => ['date_from' => $date_from, 'date_to' => $date_to],
            'filters' => [
                'limit' => $limit,
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'q' => ($q === '' ? null : $q),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
                'brand_scope' => $brand_scope,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR'
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
