<?php
// /api/brands/rank.php
// Page2 - Chart2: "Your Rank" (Top6 + optional selected brand in same chart)

define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

$t0 = microtime(true);

$user  = auth_user();
$scope = require_role_scope_filters($user);

$conn = db_connect();

// ---------- params ----------
$date_from = parse_date(get_param('date_from'));
$date_to   = parse_date(get_param('date_to')); // inclusive

// default: last 90 days
if (!$date_from || !$date_to) {
    $date_to = date('Y-m-d');
    $date_from = date('Y-m-d', strtotime('-89 days'));
}
$date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

$limit = (int)get_param('limit', 6);
if ($limit <= 0 || $limit > 20) $limit = 6;

$l1_id = get_param('l1_id'); // optional

$province = trim((string)get_param('province', ''));
$city     = trim((string)get_param('city', ''));
$area     = trim((string)get_param('area', ''));

$q = trim((string)get_param('q', '')); // brand search text

$sale_type_id = get_param('sale_type_id'); // optional

$brand_id = get_param('brand_id'); // optional

// role scope
if (!empty($scope['province'])) $province = $scope['province'];
if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id']; // brand_manager enforced

$brand_id = ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null;
$l1_id = ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null;
$sale_type_id = ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null;

// ---------- table existence checks ----------
function table_exists($conn, string $fullName): bool {
    $stmt = sqlsrv_query($conn, "SELECT CASE WHEN OBJECT_ID(?, 'U') IS NULL THEN 0 ELSE 1 END AS ok", [$fullName]);
    if ($stmt === false) return false;
    $r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);
    return (int)($r['ok'] ?? 0) === 1;
}

$hasDimBrand = table_exists($conn, 'BI.DimBrand');
$hasDimStore = table_exists($conn, 'BI.DimStore');

// اگر فیلتر شهر/منطقه/استان داده ولی DimStore نداری، بی‌راهه نریم
if (($province !== '' || $city !== '' || $area !== '') && !$hasDimStore) {
    fail('DimStore not found (BI.DimStore). Cannot filter by province/city/area.', 500);
}

// اگر سرچ برند دادی ولی DimBrand نداری، سرچ ممکن نیست
if ($q !== '' && !$hasDimBrand) {
    fail('DimBrand not found (BI.DimBrand). Cannot search by q (brand title).', 500);
}

// ---------- build joins + where ----------
$joinStore = $hasDimStore ? "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID" : "";
$joinBrand = $hasDimBrand ? "JOIN BI.DimBrand db ON db.BrandID = a.BrandID" : "";

$where  = "WHERE a.DateKey >= ? AND a.DateKey < ?";
$params = [$date_from, $date_to_excl];

if ($l1_id !== null) { $where .= " AND a.L1ID = ?"; $params[] = $l1_id; }
if ($sale_type_id !== null) { $where .= " AND a.SaleTypeID = ?"; $params[] = $sale_type_id; }

if ($hasDimStore) {
    if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
}

if ($hasDimBrand && $q !== '') {
    $where .= " AND db.BrandTitle LIKE ?";
    $params[] = '%' . $q . '%';
}

// ---------- total sales (for share percent) ----------
$sqlTotal = "
SELECT SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales
FROM BI.AggDailyStoreBrandL1 a
$joinStore
$joinBrand
$where
";
$stmtT = sqlsrv_query($conn, $sqlTotal, $params);
if ($stmtT === false) fail('DB error (rank total)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
$rt = sqlsrv_fetch_array($stmtT, SQLSRV_FETCH_ASSOC) ?: [];
$total_sales = (string)($rt['total_sales'] ?? '0');
$totalNum = (float)$total_sales;

// ---------- top6 brands (IDs + maybe title) ----------
$sqlTop = "
SELECT TOP ($limit)
    a.BrandID AS brand_id,
    " . ($hasDimBrand ? "db.BrandTitle AS brand_title," : "CAST(NULL AS nvarchar(255)) AS brand_title,") . "
    SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales
FROM BI.AggDailyStoreBrandL1 a
$joinStore
$joinBrand
$where
GROUP BY a.BrandID " . ($hasDimBrand ? ", db.BrandTitle" : "") . "
ORDER BY sales DESC
";
$stmt = sqlsrv_query($conn, $sqlTop, $params);
if ($stmt === false) fail('DB error (rank top6)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

$items = [];
$topIds = [];

while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
    $bid = (int)$r['brand_id'];
    $topIds[] = $bid;

    $salesStr = (string)($r['sales'] ?? '0');
    $salesNum = (float)$salesStr;

    $items[] = [
        'brand_id' => $bid,
        'brand_title' => $hasDimBrand ? (string)$r['brand_title'] : null,
        'sales' => $salesStr,
        'share_percent' => ($totalNum > 0 ? round(($salesNum / $totalNum) * 100, 2) : 0.0),
        'is_selected' => ($brand_id !== null && $bid === $brand_id) ? 1 : 0,
    ];
}

// ---------- selected brand (if provided) ----------
if ($brand_id !== null) {
    $inTop = in_array($brand_id, $topIds, true);

    if (!$inTop) {
        // sales of selected brand within same filters
        $sqlSel = "
        SELECT
            a.BrandID AS brand_id,
            SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales
        FROM BI.AggDailyStoreBrandL1 a
        $joinStore
        WHERE a.DateKey >= ? AND a.DateKey < ?
        " . ($l1_id !== null ? " AND a.L1ID = ? " : "") . "
        " . ($sale_type_id !== null ? " AND a.SaleTypeID = ? " : "") . "
        " . ($hasDimStore && $province !== '' ? " AND EXISTS (SELECT 1 FROM BI.DimStore ds2 WHERE ds2.StoreID=a.StoreID AND ds2.Province=?) " : "") . "
        " . ($hasDimStore && $city !== '' ? " AND EXISTS (SELECT 1 FROM BI.DimStore ds3 WHERE ds3.StoreID=a.StoreID AND ds3.City=?) " : "") . "
        " . ($hasDimStore && $area !== '' ? " AND EXISTS (SELECT 1 FROM BI.DimStore ds4 WHERE ds4.StoreID=a.StoreID AND ds4.Area=?) " : "") . "
          AND a.BrandID = ?
        GROUP BY a.BrandID
        ";

        $p = [$date_from, $date_to_excl];
        if ($l1_id !== null) $p[] = $l1_id;
        if ($sale_type_id !== null) $p[] = $sale_type_id;
        if ($hasDimStore && $province !== '') $p[] = $province;
        if ($hasDimStore && $city !== '') $p[] = $city;
        if ($hasDimStore && $area !== '') $p[] = $area;
        $p[] = $brand_id;

        $stSel = sqlsrv_query($conn, $sqlSel, $p);
        if ($stSel === false) fail('DB error (rank selected)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

        $selRow = sqlsrv_fetch_array($stSel, SQLSRV_FETCH_ASSOC);
        if ($selRow) {
            $salesStr = (string)($selRow['sales'] ?? '0');
            $salesNum = (float)$salesStr;

            $items[] = [
                'brand_id' => $brand_id,
                'brand_title' => null, // fill below
                'sales' => $salesStr,
                'share_percent' => ($totalNum > 0 ? round(($salesNum / $totalNum) * 100, 2) : 0.0),
                'is_selected' => 1,
                'added_to_compare' => 1,
            ];
        }
    }
}

// ---------- fill missing brand_title if DimBrand exists ----------
if ($hasDimBrand) {
    // put titles for any null titles
    $need = [];
    foreach ($items as $it) {
        if (empty($it['brand_title'])) $need[] = (int)$it['brand_id'];
    }
    $need = array_values(array_unique($need));
    if (count($need) > 0) {
        $in = implode(',', array_fill(0, count($need), '?'));
        $stmtB = sqlsrv_query($conn, "SELECT BrandID, BrandTitle FROM BI.DimBrand WHERE BrandID IN ($in)", $need);
        if ($stmtB !== false) {
            $map = [];
            while ($br = sqlsrv_fetch_array($stmtB, SQLSRV_FETCH_ASSOC)) {
                $map[(int)$br['BrandID']] = (string)$br['BrandTitle'];
            }
            foreach ($items as &$it) {
                $bid = (int)$it['brand_id'];
                if (empty($it['brand_title']) && isset($map[$bid])) $it['brand_title'] = $map[$bid];
            }
            unset($it);
        }
    }
}

// ---------- final sort (sales desc) ----------
usort($items, function($a, $b){
    $sa = (float)($a['sales'] ?? 0);
    $sb = (float)($b['sales'] ?? 0);
    if ($sa === $sb) return ((int)($b['is_selected'] ?? 0) <=> (int)($a['is_selected'] ?? 0));
    return $sb <=> $sa;
});

json_response([
    'data' => [
        'total_sales' => $total_sales,
        'items' => $items,
    ],
    'meta' => [
        'ok' => true,
        'role' => $user['Role'] ?? null,
        'scope' => [
            'province' => $scope['province'] ?? null,
            'brand_id' => $scope['brand_id'] ?? null,
        ],
        'filters' => [
            'date_from' => $date_from,
            'date_to' => $date_to,
            'limit' => $limit,
            'brand_id' => $brand_id,
            'l1_id' => $l1_id,
            'q' => $q ?: null,
            'province' => $province ?: null,
            'city' => $city ?: null,
            'area' => $area ?: null,
            'sale_type_id' => $sale_type_id,
        ],
        'source' => 'BI.AggDailyStoreBrandL1',
        'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
        'amount_unit' => 'IRR',
    ]
]);
