<?php
// /api/areas/sales-summary.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $conn = db_connect();

    // ---- dates (required) ----
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if (!$date_from || !$date_to) {
        fail("Missing/invalid date_from/date_to", 400);
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // ---- filters ----
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', '')); // اگر کاربر خواست یک منطقه خاص

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id']) $brand_id = $scope['brand_id'];

    // group_by: province|city|area
    $group_by = strtolower(trim((string)get_param('group_by', '')));
    if ($group_by === '') {
        // دیفالت هوشمند:
        // اگر شهر تهران انتخاب شد => area
        // اگر شهر انتخاب شد => city
        // در غیر اینصورت => province
        if ($city !== '' && (mb_strpos($city, 'تهران') !== false || strtolower($city) === 'tehran')) {
            $group_by = 'area';
        } elseif ($city !== '') {
            $group_by = 'city';
        } else {
            $group_by = 'province';
        }
    }
    if (!in_array($group_by, ['province','city','area'], true)) {
        fail("Invalid group_by (province|city|area)", 400);
    }

    // ---- base query on AggDailyStoreBrandL1 + DimStore for geography ----
    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND a.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }

    // انتخاب ستون گروه‌بندی
    if ($group_by === 'province') {
        $keyExpr = "ISNULL(NULLIF(LTRIM(RTRIM(ds.Province)), ''), N'نامشخص')";
        $titleExpr = $keyExpr;
    } elseif ($group_by === 'city') {
        $keyExpr = "ISNULL(NULLIF(LTRIM(RTRIM(ds.City)), ''), N'نامشخص')";
        $titleExpr = $keyExpr;
    } else { // area
        // مناطق تهران
        $keyExpr = "ISNULL(NULLIF(LTRIM(RTRIM(ds.Area)), ''), N'نامشخص')";
        $titleExpr = $keyExpr;
    }

    $sql = "
        SELECT
            $keyExpr  AS area_key,
            $titleExpr AS area_title,
            CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS varchar(50)) AS sales
        FROM BI.AggDailyStoreBrandL1 a
        JOIN BI.DimStore ds ON ds.StoreID = a.StoreID
        $where
        GROUP BY $keyExpr, $titleExpr
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail('DB error (areas sales-summary)', 500);

    // total
    $rows = [];
    $total = '0';

    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $sales = (string)($r['sales'] ?? '0');
        $rows[] = [
            'key' => (string)($r['area_key'] ?? 'نامشخص'),
            'title' => (string)($r['area_title'] ?? 'نامشخص'),
            'sales' => $sales
        ];

        if (function_exists('bcadd')) $total = bcadd($total, $sales, 0);
        else $total = (string)((float)$total + (float)$sales);
    }

    // percent with display + raw
    $items = [];
    foreach ($rows as $r) {
        $pct = 0.0;
        if ($total !== '0') $pct = ((float)$r['sales'] / (float)$total) * 100.0;

        $pct_raw = round($pct, 4);
        $pct_display = round($pct, 2);
        if ($pct_display == 0.0 && (float)$r['sales'] > 0) $pct_display = 0.01;

        $items[] = [
            'key' => $r['key'],
            'title' => $r['title'],
            'sales' => $r['sales'],
            'share_percent' => $pct_display,
            'share_percent_raw' => $pct_raw,
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'group_by' => $group_by,
            'total_sales' => $total,
            'items' => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'filters' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
                'group_by' => $group_by,
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
