<?php
// /src/helpers/Auth.php

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/app.php';
require_once __DIR__ . '/helperFunctions.php';

function app_config(): array {
    static $cfg;
    if ($cfg) return $cfg;
    $cfg = require __DIR__ . '/../config/app.php';
    return $cfg;
}

/**
 * برگرداندن کاربر فعلی.
 * - وقتی auth_enabled=false باشد: یک کاربر dev با نقش admin برمی‌گرداند (برای توسعه سریع)
 * - وقتی auth_enabled=true باشد: از Bearer Token می‌خواند
 */
function auth_user(): array {
    $cfg = app_config();

    if (empty($cfg['auth_enabled'])) {
        return [
            'UserID' => 0,
            'Username' => 'dev',
            'Role' => 'admin',
            'Province' => null,
            'BrandID' => null,
            'BrandTitle' => null,
            'IsActive' => 1,
        ];
    }

    $token = bearer_token();
    if (!$token) fail('Unauthorized (missing token)', 401);

    $conn = db_connect();

    $sql = "
        SELECT TOP 1
            u.UserID, u.Username, u.Role, u.Province, u.BrandID, u.BrandTitle, u.IsActive
        FROM BI.UserTokens t
        JOIN BI.Users u ON u.UserID = t.UserID
        WHERE t.Token = ? AND (t.ExpiresAt IS NULL OR t.ExpiresAt > SYSDATETIME())
    ";

    $stmt = sqlsrv_query($conn, $sql, [$token]);
    if ($stmt === false) fail('DB error (auth)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);
    if (!$row) fail('Unauthorized (invalid token)', 401);
    if (!(int)$row['IsActive']) fail('User inactive', 403);

    return $row;
}

/**
 * نقش‌ها:
 * - admin: دسترسی کامل
 * - province_manager: فقط همان استان
 * - brand_manager: فقط همان برند
 */
function require_role_scope_filters(array $user): array {
    $scope = [
        'role' => $user['Role'],
        'province' => null,
        'brand_id' => null,
    ];

    if ($user['Role'] === 'province_manager') {
        $scope['province'] = $user['Province'] ?: null;
        if (!$scope['province']) fail('Province scope missing for province_manager', 500);
    }

    if ($user['Role'] === 'brand_manager') {
        $scope['brand_id'] = $user['BrandID'] ? (int)$user['BrandID'] : null;
        if (!$scope['brand_id']) fail('Brand scope missing for brand_manager', 500);
    }

    return $scope;
}
