<?php
// /api/stores/top20.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

function date_add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // ---- dates ----
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // اگر تاریخ نداد، دیفالت: ماه جاری (از اولین روز ماه تا امروز/آخرین دیتای Agg)
    if (!$date_to) {
        $stmtMax = sqlsrv_query($conn, "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1");
        if ($stmtMax === false) sqlsrv_fail("DB error (max date)", 500);
        $mx = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC)['mx'] ?? null;
        if (!$mx) fail("No data to determine date range", 400);
        $date_to = (string)$mx;
    }
    if (!$date_from) {
        $date_from = date('Y-m-01', strtotime($date_to));
    }
    $date_to_excl = date_add_days($date_to, 1);

    // ---- filters ----
    $l1_id = get_param('l1_id');          // Agg: a.L1ID
    $brand_id = get_param('brand_id');    // Agg: a.BrandID
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id']) $brand_id = (string)$scope['brand_id'];

    $limit = (int)get_param('limit', 20);
    if ($limit <= 0 || $limit > 100) $limit = 20;

    // ---- build where ----
    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($l1_id !== null && $l1_id !== '') { $where .= " AND a.L1ID = ?"; $params[] = (int)$l1_id; }
    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }

    // ---- query ----
    $sql = "
        SELECT TOP ($limit)
            ds.StoreID AS store_id,
            ISNULL(NULLIF(LTRIM(RTRIM(ds.StoreName)),''), N'نامشخص') AS store_name,
            ds.YaranCode AS yaran_code,
            ds.YaranName AS yaran_name,
            ds.Province AS province,
            ds.City AS city,
            ds.Area AS area,
            CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS varchar(50)) AS sales
        FROM BI.AggDailyStoreBrandL1 a
        JOIN BI.DimStore ds ON ds.StoreID = a.StoreID
        $where
        GROUP BY ds.StoreID, ds.StoreName, ds.YaranCode, ds.YaranName, ds.Province, ds.City, ds.Area
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail("DB error (stores top20)", 500);

    $items = [];
    $rank = 0;
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $rank++;
        $items[] = [
            'rank' => $rank,
            'store_id' => (int)$r['store_id'],
            'store_name' => (string)$r['store_name'],
            'yaran_code' => $r['yaran_code'] === null ? null : (string)$r['yaran_code'],
            'yaran_name' => $r['yaran_name'] === null ? null : (string)$r['yaran_name'],
            'province' => $r['province'] === null ? null : (string)$r['province'],
            'city' => $r['city'] === null ? null : (string)$r['city'],
            'area' => $r['area'] === null ? null : (string)$r['area'],
            'sales' => (string)($r['sales'] ?? '0'),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'range' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
            ],
            'filters' => [
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
                'limit' => $limit,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
