<?php
// /api/sales/online_vs_inperson.php
// نمودار 2 صفحه 2: تفکیک فروش آنلاین / حضوری (حضوری = SaleTypeID 4)

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

$t0 = microtime(true);

$user = auth_user();
require_role($user, ['admin', 'manager', 'brand_manager']);

$date_from = parse_date(get_param('date_from'));
$date_to   = parse_date(get_param('date_to')); // inclusive

if (!$date_from || !$date_to) {
    fail('date_from و date_to الزامی است (YYYY-MM-DD).', 400);
}

$to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

$brand_id = get_param('brand_id');  // optional
$l1_id    = get_param('l1_id');     // optional
$store_id = get_param('store_id');  // optional

// اسکوپ نقش‌ها (اگر brand_manager بود brand_id رو enforce می‌کنیم)
require_role_scope_filters($user, [
    'brand_id' => $brand_id,
    'store_id' => $store_id,
]);

$db = get_db();

// Query
$sql = "
SELECT
    a.DateKey AS [date],
    SUM(CASE WHEN a.SaleTypeID = 4 THEN a.SalesAmount ELSE CAST(0 AS DECIMAL(38,0)) END) AS in_person_sales,
    SUM(CASE WHEN a.SaleTypeID <> 4 THEN a.SalesAmount ELSE CAST(0 AS DECIMAL(38,0)) END) AS online_sales,
    SUM(a.SalesAmount) AS total_sales
FROM BI.AggDailyStoreBrandL1 a
WHERE a.DateKey >= ? AND a.DateKey < ?
";

$params = [$date_from, $to_excl];

// فیلترهای اختیاری
if ($brand_id !== null && $brand_id !== '') {
    $sql .= " AND a.BrandID = ? ";
    $params[] = (int)$brand_id;
}
if ($l1_id !== null && $l1_id !== '') {
    $sql .= " AND a.L1ID = ? ";
    $params[] = (int)$l1_id;
}
if ($store_id !== null && $store_id !== '') {
    $sql .= " AND a.StoreID = ? ";
    $params[] = (int)$store_id;
}

$sql .= "
GROUP BY a.DateKey
ORDER BY a.DateKey;
";

$stmt = sqlsrv_query($db, $sql, $params);
if ($stmt === false) {
    fail('DB error (online_vs_inperson)', 500, [
        'sqlsrv_errors' => sqlsrv_errors(),
    ]);
}

$items = [];
$sumOnline = 0;
$sumInPerson = 0;
$sumTotal = 0;

while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
    $online = (string)($r['online_sales'] ?? '0');
    $inperson = (string)($r['in_person_sales'] ?? '0');
    $total = (string)($r['total_sales'] ?? '0');

    // جمع کل برای درصد کلی
    $sumOnline += (float)$online;
    $sumInPerson += (float)$inperson;
    $sumTotal += (float)$total;

    // درصد روزانه (اختیاری ولی مفید برای میله‌ای درصدی)
    $dayTotal = (float)$total;
    $onlinePct = $dayTotal > 0 ? round(((float)$online / $dayTotal) * 100, 2) : 0.0;
    $inPersonPct = $dayTotal > 0 ? round(((float)$inperson / $dayTotal) * 100, 2) : 0.0;

    $items[] = [
        'date' => $r['date'],
        'online_sales' => $online,
        'in_person_sales' => $inperson,
        'total_sales' => $total,
        'online_share_percent' => $onlinePct,
        'in_person_share_percent' => $inPersonPct,
    ];
}

$overall_online_pct = $sumTotal > 0 ? round(($sumOnline / $sumTotal) * 100, 2) : 0.0;
$overall_inperson_pct = $sumTotal > 0 ? round(($sumInPerson / $sumTotal) * 100, 2) : 0.0;

json_response([
    'data' => [
        'items' => $items,
        'totals' => [
            'online_sales' => (string)$sumOnline,
            'in_person_sales' => (string)$sumInPerson,
            'total_sales' => (string)$sumTotal,
            'online_share_percent' => $overall_online_pct,
            'in_person_share_percent' => $overall_inperson_pct,
        ],
        'mapping' => [
            'in_person_sale_type_id' => 4,
            'online_sale_type_rule' => 'SaleTypeID <> 4',
        ],
    ],
    'meta' => [
        'ok' => true,
        'role' => $user['role'] ?? null,
        'filters' => [
            'date_from' => $date_from,
            'date_to' => $date_to,
            'brand_id' => $brand_id ? (int)$brand_id : null,
            'l1_id' => $l1_id ? (int)$l1_id : null,
            'store_id' => $store_id ? (int)$store_id : null,
        ],
        'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
        'amount_unit' => 'IRR',
    ],
]);
