<?php
// /api/brands/top.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $conn = db_connect();

    // ---- dates ----
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        fail("Missing/invalid date_from/date_to", 400);
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // ---- filters ----
    $l1_id = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));
    $store_id = get_param('store_id'); // اگر خواستی برای صفحه ۳ هم از همین استفاده کنی

    // role scope
    // مدیر استان => province قفل
    if ($scope['province']) $province = $scope['province'];

    // مدیر برند => فقط برند خودش (این endpoint در اون حالت معنی نداره، ولی برای سازگاری فیلتر می‌کنیم)
    $forced_brand_id = $scope['brand_id'] ? (int)$scope['brand_id'] : null;

    // limit
    $limit = (int)get_param('limit', 5);
    if ($limit <= 0 || $limit > 50) $limit = 5;

    // ---- build filters ----
    $joinStore = "";
    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($l1_id !== null && $l1_id !== '') {
        $where .= " AND a.L1ID = ?";
        $params[] = (int)$l1_id;
    }

    if ($sale_type_id !== null && $sale_type_id !== '') {
        $where .= " AND a.SaleTypeID = ?";
        $params[] = (int)$sale_type_id;
    }

    if ($forced_brand_id !== null) {
        $where .= " AND a.BrandID = ?";
        $params[] = $forced_brand_id;
    }

    // فیلتر جغرافیایی/فروشگاه
    if ($province !== '' || $city !== '' || $area !== '' || ($store_id !== null && $store_id !== '')) {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
        if ($store_id !== null && $store_id !== '') { $where .= " AND ds.StoreID = ?"; $params[] = (int)$store_id; }
    }

    // ---- query ----
    // نکته: SalesAmount تو Agg می‌تونه خیلی بزرگ باشه => decimal(38,0)
    $sql = "
        SELECT TOP ($limit)
            a.BrandID AS brand_id,
            ISNULL(db.BrandTitle, CONCAT(N'Brand #', a.BrandID)) AS brand_title,
            CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS varchar(50)) AS sales
        FROM BI.AggDailyStoreBrandL1 a
        LEFT JOIN BI.DimBrand db ON db.BrandID = a.BrandID
        $joinStore
        $where
        GROUP BY a.BrandID, db.BrandTitle
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail('DB error (brands top)', 500);

    $rows = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $rows[] = [
            'brand_id' => (int)$r['brand_id'],
            'brand_title' => (string)$r['brand_title'],
            'sales' => (string)($r['sales'] ?? '0'),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'items' => $rows
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'filters' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
                'limit' => $limit,
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'store_id' => ($store_id === '' ? null : $store_id),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
                'brand_scope' => $forced_brand_id, // اگر مدیر برند بود
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
