<?php
// /api/kpi/summary.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

try {
    $t0 = microtime(true);

    $user = auth_user();
    $scope = require_role_scope_filters($user);

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // default: last 30 days
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-29 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // optional filters
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $city     = trim((string)get_param('city', ''));
    $province = trim((string)get_param('province', ''));
    $area     = trim((string)get_param('area', ''));
    $store_id = get_param('store_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id']) $brand_id = $scope['brand_id'];

    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($store_id !== null && $store_id !== '') { $where .= " AND a.StoreID = ?"; $params[] = (int)$store_id; }
    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND a.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    $joinStore = "";
    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
    }

    $conn = db_connect();

    // نکته: بعضی رکوردها SalesAmount بسیار بزرگ دارند (بالاتر از bigint)
    // پس قبل از SUM تبدیل به DECIMAL بزرگ می‌کنیم تا overflow ندهد.
    $sqlAgg = "
        SELECT
            SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales,
            COUNT(DISTINCT a.StoreID) AS stores_count,
            COUNT(DISTINCT a.BrandID) AS brands_count,
            COUNT(DISTINCT a.L1ID)    AS l1_count,
            CAST(
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount))
                / NULLIF(COUNT(DISTINCT a.StoreID), 0)
                AS decimal(38,0)
            ) AS avg_sales_per_store
        FROM BI.AggDailyStoreBrandL1 a
        $joinStore
        $where
    ";

    $stmt = sqlsrv_query($conn, $sqlAgg, $params);
    if ($stmt === false) fail('DB error (kpi agg)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC) ?: [];
    // total_sales ممکن است بزرگ‌تر از int/PHP باشد → string نگه می‌داریم
    $total_sales  = (string)($r['total_sales'] ?? '0');
    $stores_count = (int)($r['stores_count'] ?? 0);
    $brands_count = (int)($r['brands_count'] ?? 0);
    $l1_count     = (int)($r['l1_count'] ?? 0);

    $avg_sales = (string)($r['avg_sales_per_store'] ?? '0');

    // last week sales (relative to date_to)
    $week_from = date('Y-m-d', strtotime($date_to_excl . ' -7 days'));

    $sqlWeek = "
        SELECT SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS week_sales
        FROM BI.AggDailyStoreBrandL1 a
        $joinStore
        $where
          AND a.DateKey >= ? AND a.DateKey < ?
    ";
    $paramsWeek = $params;
    $paramsWeek[] = $week_from;
    $paramsWeek[] = $date_to_excl;

    $stmtW = sqlsrv_query($conn, $sqlWeek, $paramsWeek);
    if ($stmtW === false) fail('DB error (kpi week)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rw = sqlsrv_fetch_array($stmtW, SQLSRV_FETCH_ASSOC) ?: [];
    $week_sales = (string)($rw['week_sales'] ?? '0');

    // products_count (needs SaleDetail)
    $sqlProd = "
        SELECT APPROX_COUNT_DISTINCT(ProductID) AS products_count
        FROM BI.SaleDetail sd
        WHERE sd.FactorDate >= ? AND sd.FactorDate < ?
    ";
    $p2 = [$date_from, $date_to_excl];

    if ($store_id !== null && $store_id !== '') { $sqlProd .= " AND sd.StoreID = ?"; $p2[] = (int)$store_id; }
    if ($brand_id !== null && $brand_id !== '') { $sqlProd .= " AND sd.BrandID = ?"; $p2[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $sqlProd .= " AND sd.ProductGroupID = ?"; $p2[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $sqlProd .= " AND sd.SaleTypeID = ?"; $p2[] = (int)$sale_type_id; }

    if ($province !== '') { $sqlProd .= " AND sd.Province = ?"; $p2[] = $province; }
    if ($city !== '')     { $sqlProd .= " AND sd.City = ?";     $p2[] = $city; }
    if ($area !== '')     { $sqlProd .= " AND sd.Area = ?";     $p2[] = $area; }

    $stmtP = sqlsrv_query($conn, $sqlProd, $p2);
    if ($stmtP === false) fail('DB error (kpi products)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rp = sqlsrv_fetch_array($stmtP, SQLSRV_FETCH_ASSOC) ?: [];
    $products_count = (int)($rp['products_count'] ?? 0);

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'products_count' => $products_count,
            'brands_count'   => $brands_count,
            'categories_l1_count' => $l1_count,
            'stores_count'   => $stores_count,
            'total_sales'    => $total_sales,
            'avg_sales_per_store' => $avg_sales,
            'last_week_sales' => $week_sales,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
                'brand_id' => $brand_id !== '' ? ($brand_id !== null ? (int)$brand_id : null) : null,
                'l1_id' => $l1_id !== '' ? ($l1_id !== null ? (int)$l1_id : null) : null,
                'province' => $province ?: null,
                'city' => $city ?: null,
                'area' => $area ?: null,
                'store_id' => $store_id !== '' ? ($store_id !== null ? (int)$store_id : null) : null,
                'sale_type_id' => $sale_type_id !== '' ? ($sale_type_id !== null ? (int)$sale_type_id : null) : null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
