<?php
// /api/stores/selected/l2_sales_top10.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') fail('store_id is required', 400);
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $limit = (int)get_param('limit', 10);
    if ($limit <= 0) $limit = 10;
    if ($limit > 50) $limit = 50;

    $conn = db_connect();

    // province manager scope check
    if (!empty($scope['province'])) {
        $stChk = sqlsrv_query($conn, "SELECT TOP 1 Province FROM BI.DimStore WHERE StoreID = ?", [$store_id]);
        if ($stChk === false) fail('DB error (store scope check)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        $rChk = sqlsrv_fetch_array($stChk, SQLSRV_FETCH_ASSOC);
        if (!$rChk || trim((string)$rChk['Province']) !== trim((string)$scope['province'])) {
            json_response(['data' => [], 'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province']]);
        }
    }

    $where = "WHERE sd.StoreID = ? AND sd.FactorDate >= ? AND sd.FactorDate < ?";
    $params = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND sd.ProductGroupID = ?"; $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND sd.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // 1) total sales
    $sqlTotal = "
        SELECT CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice))) AS total_sales
        FROM BI.SaleDetail sd WITH (NOLOCK)
        $where
    ";
    $stT = sqlsrv_query($conn, $sqlTotal, $params);
    if ($stT === false) fail('DB error (l2 total)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sqlTotal]);

    $rt = sqlsrv_fetch_array($stT, SQLSRV_FETCH_ASSOC) ?: [];
    $total_sales = (string)($rt['total_sales'] ?? '0');

    // 2) top L2
    $sqlTop = "
        SELECT TOP ($limit)
            sd.ProductCategoryID AS l2_id,
            MAX(sd.ProductCategoryTitle) AS l2_title,
            CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice))) AS sales_amount
        FROM BI.SaleDetail sd WITH (NOLOCK)
        $where
          AND sd.ProductCategoryID IS NOT NULL
        GROUP BY sd.ProductCategoryID
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
    ";
    $st = sqlsrv_query($conn, $sqlTop, $params);
    if ($st === false) fail('DB error (l2 top10)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sqlTop]);

    // safe percent calc
    $total_num = (float)preg_replace('/[^0-9\-]/', '', $total_sales);
    if ($total_num <= 0) $total_num = 0.0;

    $items = [];
    while ($r = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC)) {
        $amt_str = (string)($r['sales_amount'] ?? '0');
        $amt_num = (float)preg_replace('/[^0-9\-]/', '', $amt_str);
        $share = ($total_num > 0) ? round(($amt_num / $total_num) * 100, 2) : 0;

        $items[] = [
            'l2_id' => (int)$r['l2_id'],
            'l2_title' => (string)($r['l2_title'] ?? ''),
            'sales_amount' => $amt_str,
            'share_percent' => $share,
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'total_sales' => $total_sales,
            'items' => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'store_id' => $store_id,
                'date_from' => $date_from,
                'date_to' => $date_to,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id' => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
                'limit' => $limit,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.SaleDetail (Store-scoped L2 Top)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
