<?php
// /api/sales/weekly_pattern.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}
function date_add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // ---- dates ----
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // default: ماه جاری بر اساس آخرین تاریخ موجود در Agg
    if (!$date_to) {
        $stmtMax = sqlsrv_query($conn, "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1");
        if ($stmtMax === false) sqlsrv_fail('DB error (max date)', 500);
        $mx = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC)['mx'] ?? null;
        if (!$mx) fail("No data to determine date range", 400);
        $date_to = (string)$mx;
    }
    if (!$date_from) {
        $date_from = date('Y-m-01', strtotime($date_to));
    }
    $date_to_excl = date_add_days($date_to, 1);

    // ---- filters ----
    $l1_id = get_param('l1_id');          // a.L1ID
    $brand_id = get_param('brand_id');    // a.BrandID
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // role scope
    if (!empty($scope['province'])) $province = $scope['province'];
    if (!empty($scope['brand_id'])) $brand_id = (string)$scope['brand_id'];

    // ---- build where ----
    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($l1_id !== null && $l1_id !== '') { $where .= " AND a.L1ID = ?"; $params[] = (int)$l1_id; }
    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // DimStore filters
    if ($province !== '' || $city !== '' || $area !== '') {
        $where .= " AND a.StoreID IN (SELECT StoreID FROM BI.DimStore ds WHERE 1=1";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
        $where .= ")";
    }

    // ---- query: avg sales per weekday ----
    // weekday: شنبه=0 ... جمعه=6 (برای فرانت راحت)
    $sql = "
        WITH daily AS (
            SELECT
                a.DateKey,
                CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS decimal(38,0)) AS sales
            FROM BI.AggDailyStoreBrandL1 a
            $where
            GROUP BY a.DateKey
        )
        SELECT
            ((DATEPART(WEEKDAY, d.DateKey) + @@DATEFIRST - 2) % 7) AS weekday_idx,
            CAST(AVG(CAST(d.sales AS decimal(38,2))) AS decimal(38,2)) AS avg_sales,
            COUNT(1) AS days_count
        FROM daily d
        GROUP BY ((DATEPART(WEEKDAY, d.DateKey) + @@DATEFIRST - 2) % 7)
        ORDER BY weekday_idx
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail("DB error (weekly pattern)", 500);

    // base: 0..6 even if missing
    $names = ["شنبه","یکشنبه","دوشنبه","سه‌شنبه","چهارشنبه","پنجشنبه","جمعه"];
    $map = [];
    for ($i=0;$i<7;$i++){
        $map[$i] = ['weekday'=>$names[$i], 'weekday_idx'=>$i, 'avg_sales'=>"0", 'days_count'=>0];
    }

    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $idx = (int)$r['weekday_idx'];
        $map[$idx] = [
            'weekday' => $names[$idx],
            'weekday_idx' => $idx,
            'avg_sales' => (string)($r['avg_sales'] ?? "0"),
            'days_count' => (int)($r['days_count'] ?? 0),
        ];
    }

    $items = array_values($map);

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'range' => ['date_from'=>$date_from, 'date_to'=>$date_to],
            'filters' => [
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR'
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
