<?php
// /api/lookup/cities.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function clamp_int($v, $min, $max, $def) {
    $n = (int)$v;
    if ($n <= 0) $n = $def;
    if ($n < $min) $n = $min;
    if ($n > $max) $n = $max;
    return $n;
}

try {
    $t0 = microtime(true);
    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    $q = trim((string)get_param('q', ''));
    $limit = clamp_int(get_param('limit', 50), 1, 200, 50);

    // optional filters
    $province = trim((string)get_param('province', ''));
    $brand_id = get_param('brand_id');

    // role scope
    if (!empty($scope['province'])) $province = (string)$scope['province'];
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $where = "WHERE sd.City IS NOT NULL AND LTRIM(RTRIM(sd.City)) <> ''";
    $params = [];

    if ($province !== '') { $where .= " AND sd.Province = ?"; $params[] = $province; }
    if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($q !== '') { $where .= " AND sd.City LIKE ?"; $params[] = '%' . $q . '%'; }

    $sql = "
        SELECT TOP ($limit)
            sd.City AS city
        FROM BI.SaleDetail sd WITH (NOLOCK)
        $where
        GROUP BY sd.City
        ORDER BY sd.City ASC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) fail('DB error (lookup cities)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $city = (string)($r['city'] ?? '');
        if ($city === '') continue;
        $items[] = ['value' => $city, 'label' => $city];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => ['items' => $items],
        'meta' => [
            'ok' => true,
            'filters' => [
                'q' => $q ?: null,
                'limit' => $limit,
                'province' => $province ?: null,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
            ],
            'duration_ms' => $elapsed_ms,
            'source' => 'BI.SaleDetail (distinct City)',
        ]
    ]);
} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
