<?php
// /api/l2/top5.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

function resolve_range_or_fail(): array {
    $month_from = trim((string)get_param('month_from', ''));
    $month_to   = trim((string)get_param('month_to', ''));

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if ($month_from !== '' && $month_to !== '') {
        if (!preg_match('/^\d{4}\-\d{2}$/', $month_from) || !preg_match('/^\d{4}\-\d{2}$/', $month_to)) {
            fail('Invalid month_from/month_to. Use YYYY-MM', 400);
        }
        $date_from = $month_from . '-01';
        $date_to   = date('Y-m-t', strtotime($month_to . '-01'));
    }

    if (!$date_from || !$date_to) {
        fail('month_from/month_to (or date_from/date_to) are required', 400);
    }

    $date_to_excl = add_days($date_to, 1);
    return [$date_from, $date_to, $date_to_excl];
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // Range (ماه)
    [$date_from, $date_to, $date_to_excl] = resolve_range_or_fail();

    // Filters (page2)
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id'); // اختیاری، اگر خواستی L2ها داخل یک L1 محدود بشن

    $city = trim((string)get_param('city', ''));
    $area = trim((string)get_param('area', ''));

    // scope
    $province_scope = !empty($scope['province']) ? (string)$scope['province'] : '';
    $brand_scope    = !empty($scope['brand_id']) ? (int)$scope['brand_id'] : null;
    if ($brand_scope !== null) $brand_id = $brand_scope;

    // area فقط تهران
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    // Build WHERE
    $where  = "WHERE sd.FactorDate >= ? AND sd.FactorDate < ?";
    $params = [$date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND sd.ProductGroupID = ?"; $params[] = (int)$l1_id; }

    if ($province_scope !== '') { $where .= " AND sd.Province = ?"; $params[] = $province_scope; }
    if ($city !== '')           { $where .= " AND sd.City = ?";     $params[] = $city; }
    if ($area !== '')           { $where .= " AND sd.Area = ?";     $params[] = $area; }

    // Query: Top 5 L2 by TotalPrice
    $sql = "
        WITH base AS (
            SELECT
                sd.ProductCategoryID AS l2_id,
                MAX(sd.ProductCategoryTitle) AS l2_title,
                SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS sales
            FROM BI.SaleDetail sd WITH (NOLOCK)
            $where
              AND sd.ProductCategoryID IS NOT NULL
            GROUP BY sd.ProductCategoryID
        ),
        totals AS (
            SELECT SUM(sales) AS total_sales FROM base
        ),
        top5 AS (
            SELECT TOP (5) *
            FROM base
            ORDER BY sales DESC
        )
        SELECT
            t.l2_id,
            t.l2_title,
            CAST(t.sales AS varchar(60)) AS sales,
            CAST(
                (CAST(t.sales AS decimal(38,6)) * 100) / NULLIF(CAST(tt.total_sales AS decimal(38,6)), 0)
            AS decimal(9,2)) AS share_percent,
            CAST(tt.total_sales AS varchar(60)) AS total_sales
        FROM top5 t
        CROSS JOIN totals tt
        ORDER BY t.sales DESC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        fail('DB error (top l2)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);
    }

    $items = [];
    $total_sales = "0";

    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $total_sales = (string)($r['total_sales'] ?? '0');
        $items[] = [
            'l2_id' => (int)$r['l2_id'],
            'l2_title' => (string)($r['l2_title'] ?? ''),
            'sales' => (string)($r['sales'] ?? '0'),
            'share_percent' => (float)($r['share_percent'] ?? 0),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'items' => $items,
            'total_sales' => $total_sales,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $province_scope ?: null,
                'brand_id' => $brand_scope,
            ],
            'filters' => [
                'month_from' => get_param('month_from') ?: null,
                'month_to'   => get_param('month_to') ?: null,
                'date_from'  => $date_from,
                'date_to'    => $date_to,
                'brand_id'   => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'      => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'city'       => $city ?: null,
                'area'       => $area ?: null,
                'top'        => 5,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.SaleDetail (L2=ProductCategoryID)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
