<?php
// /api/l1/top_products.php  (REAL L1 = 12 groups)
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

$t0 = microtime(true);

$user  = auth_user();
$scope = require_role_scope_filters($user);

$conn = db_connect();
if (!$conn) fail('DB connection not available. (conn is null)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

// required
$l1_id = get_param('l1_id');
if ($l1_id === null || $l1_id === '') fail('l1_id is required', 400);
$l1_id = (int)$l1_id;
if ($l1_id < 1 || $l1_id > 12) fail('l1_id must be 1..12 (REAL L1)', 400);

// dates
$date_from = parse_date(get_param('date_from'));
$date_to   = parse_date(get_param('date_to')); // inclusive
if (!$date_from || !$date_to) {
    $date_to = date('Y-m-d');
    $date_from = date('Y-m-d', strtotime('-89 days'));
}
$date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

$limit = (int)get_param('limit', 5);
if ($limit <= 0 || $limit > 50) $limit = 5;

$brand_id = get_param('brand_id');
$province = trim((string)get_param('province', ''));
$city     = trim((string)get_param('city', ''));
$area     = trim((string)get_param('area', ''));
$store_id = get_param('store_id');
$sale_type_id = get_param('sale_type_id');

// scope
if (!empty($scope['province'])) $province = $scope['province'];
if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

$brand_id = ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null;
$store_id = ($store_id !== null && $store_id !== '') ? (int)$store_id : null;
$sale_type_id = ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null;

// join store only if needed
$joinStore = "";
$where  = "WHERE sd.FactorDate >= ? AND sd.FactorDate < ? AND sd.ProductGroupID = ?";
$params = [
    $date_from . ' 00:00:00',
    $date_to_excl . ' 00:00:00',
    $l1_id
];

if ($brand_id !== null) { $where .= " AND sd.BrandID = ?"; $params[] = $brand_id; }
if ($store_id !== null) { $where .= " AND sd.StoreID = ?"; $params[] = $store_id; }
if ($sale_type_id !== null) { $where .= " AND sd.SaleTypeID = ?"; $params[] = $sale_type_id; }

if ($province !== '' || $city !== '' || $area !== '') {
    $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = sd.StoreID";
    if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
}

// TOTAL داخل همین L1 برای درصد سهم محصولات
$sqlTotal = "
SELECT SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS total_sales
FROM BI.SaleDetail sd
$joinStore
$where
";
$stT = sqlsrv_query($conn, $sqlTotal, $params);
if ($stT === false) fail('DB error (l1 products total)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
$rt = sqlsrv_fetch_array($stT, SQLSRV_FETCH_ASSOC) ?: [];
$total_sales = (string)($rt['total_sales'] ?? '0');
$totalNum = (float)$total_sales;

// TOP PRODUCTS — بدون هیچ JOIN اضافه (ProductTitle/BrandTitle از خود SaleDetail)
$sql = "
SELECT TOP ($limit)
  sd.ProductID AS product_id,
  MAX(sd.ProductTitle) AS product_title,
  sd.BrandID AS brand_id,
  MAX(sd.BrandTitle) AS brand_title,
  SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS sales,
  SUM(TRY_CONVERT(decimal(18,3), sd.Quantity)) AS qty
FROM BI.SaleDetail sd
$joinStore
$where
GROUP BY sd.ProductID, sd.BrandID
ORDER BY sales DESC
OPTION (RECOMPILE)
";
$st = sqlsrv_query($conn, $sql, $params);
if ($st === false) fail('DB error (l1 top products)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

$items = [];
while ($r = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC)) {
    $salesStr = (string)($r['sales'] ?? '0');
    $salesNum = (float)$salesStr;
    $items[] = [
        'product_id' => (int)$r['product_id'],
        'product_title' => (string)($r['product_title'] ?? ''),
        'brand_id' => (int)($r['brand_id'] ?? 0),
        'brand_title' => (string)($r['brand_title'] ?? ''),
        'sales' => $salesStr,
        'qty' => (string)($r['qty'] ?? '0'),
        'share_percent' => ($totalNum > 0 ? round(($salesNum / $totalNum) * 100, 2) : 0.0),
    ];
}

json_response([
  'data' => [
      'l1_id' => $l1_id,
      'total_sales' => $total_sales,
      'items' => $items,
  ],
  'meta' => [
      'ok' => true,
      'role' => $user['Role'] ?? null,
      'filters' => [
          'date_from' => $date_from,
          'date_to' => $date_to,
          'l1_id' => $l1_id,
          'limit' => $limit,
          'brand_id' => $brand_id,
          'province' => $province ?: null,
          'city' => $city ?: null,
          'area' => $area ?: null,
          'store_id' => $store_id,
          'sale_type_id' => $sale_type_id,
      ],
      'source' => 'BI.SaleDetail (ProductGroupID = REAL L1 1..12)',
      'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
      'amount_unit' => 'IRR'
  ]
]);
