<?php
// /api/admin/products/search.php  (FAST: ProductMaster + Override)
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

try {
    $t0 = microtime(true);

    $user = auth_user();
    $conn = db_connect();

    $q = trim((string)get_param('q', ''));
    $limit = (int)get_param('limit', 20);
    if ($limit <= 0) $limit = 20;
    if ($limit > 50) $limit = 50;

    if ($q === '' || mb_strlen($q) < 2) {
        json_response([
            'data' => ['items' => []],
            'meta' => [
                'ok' => true,
                'note' => 'q is empty or too short (min 2 chars)',
                'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
                'source' => 'BI.ProductMaster'
            ]
        ]);
        exit;
    }

    $is_numeric = preg_match('/^\d+$/', $q) === 1;

    if ($is_numeric) {
        $sql = "
            SELECT TOP ($limit)
                pm.ProductID AS product_id,
                pm.ProductTitle AS product_title,

                COALESCE(oh.BrandID, pm.BrandID) AS brand_id,
                COALESCE(NULLIF(oh.BrandTitle,''), pm.BrandTitle) AS brand_title,

                COALESCE(oh.L1ID, pm.L1ID) AS l1_id,
                COALESCE(NULLIF(oh.L1Title,''), pm.L1Title) AS l1_title,

                COALESCE(oh.L2ID, pm.L2ID) AS l2_id,
                COALESCE(NULLIF(oh.L2Title,''), pm.L2Title) AS l2_title,

                COALESCE(oh.L3ID, pm.L3ID) AS l3_id,
                COALESCE(NULLIF(oh.L3Title,''), pm.L3Title) AS l3_title,

                CAST(CASE WHEN oa.ProductID IS NULL THEN 0 ELSE 1 END AS bit) AS is_overridden,
                oa.ActiveVersion AS override_version
            FROM BI.ProductMaster pm
            LEFT JOIN BI.ProductOverrideActive oa
                ON oa.ProductID = pm.ProductID
            LEFT JOIN BI.ProductOverrideHistory oh
                ON oh.ProductID = oa.ProductID AND oh.VersionNo = oa.ActiveVersion
            WHERE pm.ProductID = ?
            ORDER BY pm.ProductID DESC
            OPTION (RECOMPILE);
        ";
        $params = [(int)$q];
    } else {
        $sql = "
            SELECT TOP ($limit)
                pm.ProductID AS product_id,
                pm.ProductTitle AS product_title,

                COALESCE(oh.BrandID, pm.BrandID) AS brand_id,
                COALESCE(NULLIF(oh.BrandTitle,''), pm.BrandTitle) AS brand_title,

                COALESCE(oh.L1ID, pm.L1ID) AS l1_id,
                COALESCE(NULLIF(oh.L1Title,''), pm.L1Title) AS l1_title,

                COALESCE(oh.L2ID, pm.L2ID) AS l2_id,
                COALESCE(NULLIF(oh.L2Title,''), pm.L2Title) AS l2_title,

                COALESCE(oh.L3ID, pm.L3ID) AS l3_id,
                COALESCE(NULLIF(oh.L3Title,''), pm.L3Title) AS l3_title,

                CAST(CASE WHEN oa.ProductID IS NULL THEN 0 ELSE 1 END AS bit) AS is_overridden,
                oa.ActiveVersion AS override_version
            FROM BI.ProductMaster pm
            LEFT JOIN BI.ProductOverrideActive oa
                ON oa.ProductID = pm.ProductID
            LEFT JOIN BI.ProductOverrideHistory oh
                ON oh.ProductID = oa.ProductID AND oh.VersionNo = oa.ActiveVersion
            WHERE pm.ProductTitle LIKE ? OR pm.BrandTitle LIKE ?
            ORDER BY pm.ProductID DESC
            OPTION (RECOMPILE);
        ";
        $like = '%' . $q . '%';
        $params = [$like, $like];
    }

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        fail('DB error (product search)', 500, [
            'sqlsrv_errors' => sqlsrv_errors(),
        ]);
    }

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'product_id' => (int)$r['product_id'],
            'product_title' => (string)($r['product_title'] ?? ''),

            'brand_id' => $r['brand_id'] !== null ? (int)$r['brand_id'] : null,
            'brand_title' => (string)($r['brand_title'] ?? ''),

            'l1_id' => $r['l1_id'] !== null ? (int)$r['l1_id'] : null,
            'l1_title' => (string)($r['l1_title'] ?? ''),

            'l2_id' => $r['l2_id'] !== null ? (int)$r['l2_id'] : null,
            'l2_title' => (string)($r['l2_title'] ?? ''),

            'l3_id' => $r['l3_id'] !== null ? (int)$r['l3_id'] : null,
            'l3_title' => (string)($r['l3_title'] ?? ''),

            'is_overridden' => (bool)($r['is_overridden'] ?? false),
            'override_version' => $r['override_version'] !== null ? (int)$r['override_version'] : null,
        ];
    }

    json_response([
        'data' => ['items' => $items],
        'meta' => [
            'ok' => true,
            'q' => $q,
            'limit' => $limit,
            'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
            'source' => 'BI.ProductMaster (+Override)',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
