/* global Chart */

const API_BASE = '/dashboard-project/api';

function $(id){ return document.getElementById(id); }

function logLine(msg){
  const el = $('log');
  const now = new Date().toLocaleTimeString();
  el.textContent = `[${now}] ${msg}\n` + el.textContent;
}

function getToken(){
  return ($('token').value || localStorage.getItem('dashboard_token') || '').trim();
}

function setToken(t){
  $('token').value = t || '';
  if (t) localStorage.setItem('dashboard_token', t);
  else localStorage.removeItem('dashboard_token');
}

function setAuthStatus(text, ok=true){
  const el = $('authStatus');
  el.textContent = text;
  el.style.color = ok ? '#0a7a2f' : '#b00020';
}

async function apiGet(path, params={}){
  const url = new URL(API_BASE + path, window.location.origin);
  Object.entries(params).forEach(([k,v]) => {
    if (v === null || v === undefined || v === '') return;
    url.searchParams.set(k, v);
  });

  const token = getToken();
  const headers = token ? { 'Authorization': `Bearer ${token}` } : {};

  const res = await fetch(url.toString(), { headers });
  const text = await res.text();
  let json;
  try { json = JSON.parse(text); } catch(e){
    throw new Error(`Invalid JSON from ${path}: ${text.slice(0,200)}`);
  }
  if (!res.ok || (json?.meta && json.meta.ok === false)) {
    const err = json?.meta?.error || `HTTP ${res.status}`;
    throw new Error(`${path}: ${err}`);
  }
  return json;
}

function readFilters(){
  return {
    date_from: $('date_from').value,
    date_to: $('date_to').value,
    l1_id: $('l1_id').value,
    brand_id: $('brand_id').value,
    province: $('province').value,
    city: $('city').value,
    area: $('area').value,
  };
}

// ------- Charts -------
let chartTimeseries, chartWeekAvg, chartTehranAreas, chartL1Share, chartTopBrands, chartDailyShare;

function destroyChart(c){
  if (c) { try { c.destroy(); } catch(_){} }
  return null;
}

function fmtIRR(x){
  if (x === null || x === undefined) return '0';
  const s = String(x);
  const digits = s.replace(/[^0-9\-]/g,'');
  if (!digits) return s;
  const neg = digits.startsWith('-');
  const d = neg ? digits.slice(1) : digits;
  const out = d.replace(/\B(?=(\d{3})+(?!\d))/g, ',');
  return (neg?'-':'') + out;
}

function safeBigInt(v){
  if (v === null || v === undefined) return 0n;
  let s = String(v).trim();
  if (!s || s.toLowerCase() === 'null') return 0n;
  if (s.includes('.')) s = s.split('.')[0];
  s = s.replace(/[^0-9\-]/g,'');
  if (!s || s === '-') return 0n;
  try { return BigInt(s); } catch(_) { return 0n; }
}

function toBillion(v){
  // برای نمودارها عدد رو به "میلیارد" تبدیل می‌کنیم که JS overflow نکنه
  const bi = safeBigInt(v);
  return Number(bi / 1000000000n);
}

function renderKpis(k){
  const kpis = [
    {key:'products_count', title:'تعداد کل محصولات'},
    {key:'brands_count', title:'تعداد کل برندها'},
    {key:'categories_l1_count', title:'تعداد دسته‌های محصولات (L1)'},
    {key:'stores_count', title:'تعداد فروشگاه‌ها'},
    {key:'total_sales', title:'فروش کل'},
    {key:'avg_sales_per_store', title:'میانگین فروش / فروشگاه'},
    {key:'last_week_sales', title:'فروش هفته گذشته'},
  ];

  const wrap = $('kpis');
  wrap.innerHTML = '';

  kpis.forEach(it => {
    const val = k[it.key];
    const card = document.createElement('div');
    card.className = 'kpi';
    card.innerHTML = `
      <div class="kpi-title">${it.title}</div>
      <div class="kpi-value">${fmtIRR(val)}</div>
    `;
    wrap.appendChild(card);
  });
}

function renderTimeseries(ts){
  const labels = (ts.data.current || []).map(x => x.date);
  const cur  = (ts.data.current  || []).map(x => toBillion(x.sales));
  const prev = (ts.data.previous || []).map(x => toBillion(x.sales));

  chartTimeseries = destroyChart(chartTimeseries);
  chartTimeseries = new Chart($('chartTimeseries'), {
    type: 'line',
    data: {
      labels,
      datasets: [
        { label: 'Current (B)', data: cur, tension: 0.25 },
        { label: 'Previous (B)', data: prev, tension: 0.25 },
      ]
    },
    options: {
      responsive: true,
      plugins: { legend: { position: 'bottom' } },
      scales: { y: { ticks: { callback: v => `${fmtIRR(v)}B` } } }
    }
  });

  $('tsMeta').textContent =
    `range: ${ts.meta.ranges?.current?.from} → ${ts.meta.ranges?.current?.to} | prev: ${ts.meta.ranges?.previous?.from} → ${ts.meta.ranges?.previous?.to}`;
}

/** ✅ Chart 2: weekly avg pattern */
function renderWeekAvg(res){
  const items = (res.data.items || []).slice();
  items.sort((a,b) => Number(a.weekday_idx ?? 0) - Number(b.weekday_idx ?? 0));

  const labels = items.map(x => x.weekday || '');
  const values = items.map(x => toBillion(x.avg_sales));

  chartWeekAvg = destroyChart(chartWeekAvg);
  chartWeekAvg = new Chart($('chartWeekAvg'), {
    type: 'bar',
    data: { labels, datasets: [{ label: 'Avg Sales (B)', data: values }] },
    options: {
      responsive: true,
      plugins: { legend: { display: false } },
      scales: { y: { ticks: { callback: v => `${fmtIRR(v)}B` } } }
    }
  });

  const totalDays = items.reduce((s,x)=> s + Number(x.days_count || 0), 0);
  $('weekMeta').textContent = `days_used: ${totalDays}`;
}

function renderL1Share(res){
  const items = res.data.items || [];
  const labels = items.map(x => x.l1_title);
  const values = items.map(x => toBillion(x.sales));

  chartL1Share = destroyChart(chartL1Share);
  chartL1Share = new Chart($('chartL1Share'), {
    type: 'doughnut',
    data: { labels, datasets: [{ data: values }] },
    options: {
      responsive: true,
      plugins: { legend: { position: 'right' } }
    }
  });

  $('l1Meta').textContent = `total_sales: ${fmtIRR(res.data.total_sales)}`;
}

/** ✅ Chart 4: tehran areas */
function renderTehranAreas(res){
  const items = (res.data.items || []).slice();
  items.sort((a,b) => Number(a.area) - Number(b.area));

  const labels = items.map(x => `منطقه ${x.area}`);
  const values = items.map(x => toBillion(x.sales));

  chartTehranAreas = destroyChart(chartTehranAreas);
  chartTehranAreas = new Chart($('chartTehranAreas'), {
    type: 'bar',
    data: { labels, datasets: [{ label: 'Sales (B)', data: values }] },
    options: {
      responsive: true,
      plugins: { legend: { display: false } },
      scales: { y: { ticks: { callback: v => `${fmtIRR(v)}B` } } }
    }
  });

  // top area
  let top = null;
  let topVal = -1n;
  for (const it of items){
    const s = safeBigInt(it.sales);
    if (s > topVal){ topVal = s; top = it; }
  }
  $('areaMeta').textContent = top ? `Top: منطقه ${top.area} | sales: ${fmtIRR(top.sales)}` : '';
}

function renderTopBrands(res){
  const items = res.data.items || [];
  const labels = items.map(x => x.brand_title);
  const values = items.map(x => toBillion(x.sales));

  chartTopBrands = destroyChart(chartTopBrands);
  chartTopBrands = new Chart($('chartTopBrands'), {
    type: 'bar',
    data: { labels, datasets: [{ label: 'Sales (B)', data: values }] },
    options: {
      responsive: true,
      plugins: { legend: { display: false } },
      scales: { y: { ticks: { callback: v => `${fmtIRR(v)}B` } } }
    }
  });
}

function renderDailyShare(res){
  const days = res.data.days || [];
  const brands = res.data.brands || [];
  const labels = days.map(d => d.date);

  const datasets = brands.map((b) => {
    const data = days.map(d => {
      const it = (d.items || []).find(x => String(x.brand_id) === String(b.brand_id));
      return it ? Number(it.share_percent || 0) : 0;
    });
    return { label: b.brand_title || String(b.brand_id), data };
  });

  chartDailyShare = destroyChart(chartDailyShare);
  chartDailyShare = new Chart($('chartDailyShare'), {
    type: 'bar',
    data: { labels, datasets },
    options: {
      responsive: true,
      plugins: { legend: { position: 'bottom' } },
      scales: { y: { suggestedMax: 100, ticks: { callback: v => v + '%' } } }
    }
  });

  $('shareMeta').textContent = `days: ${labels.length} | brands: ${brands.length}`;
}

function renderTopStores(res){
  const body = $('topStoresBody');
  body.innerHTML = '';

  (res.data.items || []).forEach(row => {
    const tr = document.createElement('tr');
    tr.innerHTML = `
      <td>${row.rank}</td>
      <td>${row.store_id}</td>
      <td>${row.store_name || ''}</td>
      <td>${row.province || ''}</td>
      <td>${row.city || ''}</td>
      <td>${row.area || ''}</td>
      <td>${fmtIRR(row.sales)}</td>
    `;
    body.appendChild(tr);
  });

  $('storesMeta').textContent = `rows: ${(res.data.items || []).length}`;
}


// ------- AI Insights (Stage 1: rule-based) -------
function renderAIInsights(res){
  const box = $('aiInsights');
  const metaEl = $('aiMeta');
  if (!box) return;

  const data = res?.data || {};
  const summary = data.summary || '';
  const cards = Array.isArray(data.cards) ? data.cards : [];

  const sevClass = (sev) => {
    switch (String(sev || '').toLowerCase()) {
      case 'success': return 'ai-success';
      case 'warning': return 'ai-warning';
      case 'danger':  return 'ai-danger';
      default:        return 'ai-info';
    }
  };

  box.innerHTML = `
    ${summary ? `<div class="ai-summary">${summary}</div>` : ''}
    ${cards.map(c => `
      <div class="ai-card ${sevClass(c.severity)}">
        <div class="ai-card-title">${c.title || ''}</div>
        <div class="ai-card-text">${c.text || ''}</div>
      </div>
    `).join('')}
  `;

  if (metaEl) {
    const ms = res?.meta?.duration_ms;
    metaEl.textContent = ms ? `زمان تولید: ${ms}ms` : '';
  }
}

async function loadInsights(){
  const box = $('aiInsights');
  const metaEl = $('aiMeta');
  if (!box) return;

  if (!getToken()) {
    box.innerHTML = `<div class="muted">توکن نداریم؛ اول Login بزن.</div>`;
    if (metaEl) metaEl.textContent = '';
    return;
  }

  const f = readFilters();
  if (metaEl) metaEl.textContent = 'در حال تولید تحلیل...';

  try {
    const res = await apiGet('/ai/insight/dashboard.php', f);
    renderAIInsights(res);
    logLine('AI insight loaded');
  } catch(e) {
    box.innerHTML = `<div class="muted">خطا در دریافت تحلیل: ${e.message}</div>`;
    if (metaEl) metaEl.textContent = '';
    logLine('AI insight error: ' + e.message);
  }
}


// ------- Actions -------
async function doLogin(){
  const username = $('username').value.trim();
  const password = $('password').value;

  try {
    const res = await apiGet('/auth/login.php', { username, password });
    const token = res.data?.token;
    if (!token) throw new Error('No token');
    setToken(token);
    setAuthStatus(`OK: ${res.data.user.username} (${res.data.user.role})`, true);
    logLine('Login OK');
  } catch(e){
    setAuthStatus('Login failed: ' + e.message, false);
    logLine('Login failed: ' + e.message);
  }
}

function doLogout(){
  setToken('');
  setAuthStatus('Logged out', true);
}

async function loadDashboard(){
  const f = readFilters();
  if (!getToken()) {
    setAuthStatus('توکن نداریم. اول Login بزن.', false);
    return;
  }

  setAuthStatus('Loading...', true);

  try {
    // KPI
    const kpi = await apiGet('/kpi/summary.php', f);
    renderKpis(kpi.data);
    logLine('KPI loaded');

    // Chart 1 - Timeseries
    const ts = await apiGet('/sales/timeseries.php', f);
    renderTimeseries(ts);
    logLine('Timeseries loaded');

    // ✅ Chart 2 - Weekly Pattern
    const week = await apiGet('/sales/weekly_pattern.php', f);
    renderWeekAvg(week);
    logLine('Weekly pattern loaded');

    // Chart 3 - L1 Share
    const l1 = await apiGet('/categories/share.php', f);
    renderL1Share(l1);
    logLine('Categories share loaded');

    // ✅ Chart 4 - Tehran Areas
    const tehran = await apiGet('/areas/tehran_sales.php', f);
    renderTehranAreas(tehran);
    logLine('Tehran areas loaded');

    // Chart 5 - Top brands
    const topBrands = await apiGet('/brands/top.php', { ...f, limit: 5 });
    renderTopBrands(topBrands);
    logLine('Top brands loaded');

    // Chart 7 - Daily share
    const dailyShare = await apiGet('/brands/daily_share.php', { ...f });
    renderDailyShare(dailyShare);
    logLine('Daily share loaded');

    // Chart 8 - Top stores
    const stores = await apiGet('/stores/top20.php', { ...f, limit: 20 });
    renderTopStores(stores);
    logLine('Top stores loaded');

    setAuthStatus('Loaded ✅', true);

    // AI Insights (non-blocking)
    loadInsights();

  } catch(e){
    setAuthStatus('Error: ' + e.message, false);
    logLine('Error: ' + e.message);
  }
}

function boot(){
  const t = localStorage.getItem('dashboard_token');
  if (t) {
    setToken(t);
    setAuthStatus('Token restored from localStorage', true);
  } else {
    setAuthStatus('Not logged in', true);
  }

  $('btnLogin').addEventListener('click', doLogin);
  $('btnLogout').addEventListener('click', doLogout);
  $('btnLoad').addEventListener('click', loadDashboard);

  const btnIns = $('btnInsights');
  if (btnIns) btnIns.addEventListener('click', loadInsights);
}

boot();
