(() => {
  const $ = (id) => document.getElementById(id);

  const logEl = $("log");
  let lastSearchItems = [];
  let lastGet = null;

  function setLog(obj) {
    logEl.textContent = JSON.stringify(obj, null, 2);
  }

  function baseApi() {
    let v = $("baseApi").value.trim();
    if (!v) v = "/api";
    // remove trailing slash
    if (v.endsWith("/")) v = v.slice(0, -1);
    return v;
  }

  async function apiFetch(path, { method = "GET", qs = null, json = null } = {}) {
    const url = new URL(baseApi() + path, window.location.origin);
    if (qs) Object.entries(qs).forEach(([k, v]) => {
      if (v !== undefined && v !== null && String(v).trim() !== "") url.searchParams.set(k, v);
    });

    const opts = {
      method,
      credentials: "include", // IMPORTANT: use session/cookie
      headers: {}
    };

    if (json !== null) {
      opts.headers["Content-Type"] = "application/json";
      opts.body = JSON.stringify(json);
    }

    const res = await fetch(url.toString(), opts);
    const text = await res.text();
    let data;
    try { data = JSON.parse(text); } catch { data = { raw: text }; }

    if (!res.ok) {
      throw { status: res.status, data };
    }
    return data;
  }

  function badge(status) {
    if (status === "approved") return `<span class="badge ok">approved</span>`;
    if (status === "rejected") return `<span class="badge rejected">rejected</span>`;
    return `<span class="badge pending">pending</span>`;
  }

  function fmtOldNew(row) {
    const o = row.old || {};
    const n = row.new || {};
    return `
      <div class="small">brand: ${o.brand_id ?? "-"} → ${n.brand_id ?? "-"}</div>
      <div class="small">l1: ${o.l1_id ?? "-"} → ${n.l1_id ?? "-"}</div>
      <div class="small">l2: ${o.l2_id ?? "-"} → ${n.l2_id ?? "-"}</div>
      <div class="small">l3: ${o.l3_id ?? "-"} → ${n.l3_id ?? "-"}</div>
    `;
  }

  async function onPing() {
    try {
      const data = await apiFetch("/admin/products/pending.php", { qs: { status: "pending", limit: 1 } });
      setLog({ ok: true, ping: data });
    } catch (e) {
      setLog({ ok: false, where: "ping", error: e });
    }
  }

  async function onSearch() {
    const q = $("qSearch").value.trim();
    const limit = $("limitSearch").value.trim() || "10";
    try {
      const data = await apiFetch("/admin/products/search.php", { qs: { q, limit } });
      lastSearchItems = (data?.data?.items) || [];
      setLog(data);
    } catch (e) {
      setLog({ ok: false, where: "search", error: e });
    }
  }

  function useFirst() {
    if (!lastSearchItems.length) return;
    const id = lastSearchItems[0].product_id;
    $("productId").value = id;
    $("editProductId").value = id;
  }

  async function onGet() {
    const product_id = $("productId").value.trim();
    try {
      const data = await apiFetch("/admin/products/get.php", { qs: { product_id } });
      lastGet = data;
      // auto fill edit product id
      $("editProductId").value = data?.data?.product_id ?? "";
      setLog(data);
    } catch (e) {
      setLog({ ok: false, where: "get", error: e });
    }
  }

  function copyFromGet() {
    if (!lastGet?.data?.product_id) return;
    $("editProductId").value = lastGet.data.product_id;
  }

  async function onRequestEdit() {
    const payload = {
      product_id: $("editProductId").value.trim(),
      new_brand_id: $("newBrandId").value.trim(),
      new_l1_id: $("newL1Id").value.trim(),
      new_l2_id: $("newL2Id").value.trim(),
      new_l3_id: $("newL3Id").value.trim(),
      note: $("editNote").value.trim()
    };

    // پاکسازی: رشته خالی نفرست
    Object.keys(payload).forEach(k => {
      if (payload[k] === "") delete payload[k];
    });

    try {
      const data = await apiFetch("/admin/products/request_edit.php", { method: "POST", json: payload });
      setLog(data);
      // بعد از ثبت، لیست pending رو رفرش کن
      await onLoadPending();
    } catch (e) {
      setLog({ ok: false, where: "request_edit", payload, error: e });
    }
  }

  async function onApprove(edit_id) {
    const note = prompt("یادداشت تایید (اختیاری):", "approved");
    try {
      const data = await apiFetch("/admin/products/approve.php", {
        method: "POST",
        json: { edit_id, note: note ?? "" }
      });
      setLog(data);
      await onLoadPending();
    } catch (e) {
      setLog({ ok: false, where: "approve", edit_id, error: e });
    }
  }

  async function onReject(edit_id) {
    const note = prompt("دلیل رد (اختیاری):", "rejected");
    try {
      const data = await apiFetch("/admin/products/reject.php", {
        method: "POST",
        json: { edit_id, note: note ?? "" }
      });
      setLog(data);
      await onLoadPending();
    } catch (e) {
      setLog({ ok: false, where: "reject", edit_id, error: e });
    }
  }

  async function onLoadPending() {
    const status = $("pendingStatus").value;
    const q = $("pendingQ").value.trim();
    const limit = $("pendingLimit").value.trim() || "30";

    try {
      const data = await apiFetch("/admin/products/pending.php", { qs: { status, q, limit } });
      setLog(data);

      const items = data?.data?.items || [];
      const tb = $("pendingTbody");
      if (!items.length) {
        tb.innerHTML = `<tr><td colspan="6" class="small">هیچ موردی پیدا نشد.</td></tr>`;
        return;
      }

      tb.innerHTML = items.map(row => {
        const canAct = row.status === "pending";
        return `
          <tr>
            <td>${row.edit_id}<div class="small">${badge(row.status)}</div></td>
            <td>${row.product_id}</td>
            <td>${fmtOldNew(row)}</td>
            <td>${(row.note || "").replaceAll("<","&lt;")}</td>
            <td>
              <div class="small">${row.requested_by || ""}</div>
              <div class="small">${row.requested_at || ""}</div>
            </td>
            <td>
              ${canAct ? `
                <div class="actions">
                  <button data-act="approve" data-id="${row.edit_id}">Approve</button>
                  <button class="danger" data-act="reject" data-id="${row.edit_id}">Reject</button>
                </div>
              ` : `<span class="small">-</span>`}
            </td>
          </tr>
        `;
      }).join("");

      // bind buttons
      tb.querySelectorAll("button[data-act]").forEach(btn => {
        btn.addEventListener("click", () => {
          const id = parseInt(btn.getAttribute("data-id"), 10);
          const act = btn.getAttribute("data-act");
          if (act === "approve") onApprove(id);
          else onReject(id);
        });
      });

    } catch (e) {
      setLog({ ok: false, where: "pending", error: e });
    }
  }

  // Events
  $("btnPing").addEventListener("click", onPing);
  $("btnSearch").addEventListener("click", onSearch);
  $("btnUseFirst").addEventListener("click", useFirst);
  $("btnGet").addEventListener("click", onGet);
  $("btnCopyFromGet").addEventListener("click", copyFromGet);
  $("btnRequestEdit").addEventListener("click", onRequestEdit);
  $("btnLoadPending").addEventListener("click", onLoadPending);

  // initial
  setLog({ ready: true, baseApi: baseApi() });
})();
