<?php
// /api/stores/selected/product_performance_monthly.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') fail('store_id is required', 400);
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $limit = (int)get_param('limit', 6);
    if ($limit <= 0) $limit = 6;
    if ($limit > 250) $limit = 250;

    $conn = db_connect();

    // province manager scope check
    if (!empty($scope['province'])) {
        $stChk = sqlsrv_query($conn, "SELECT TOP 1 Province FROM BI.DimStore WHERE StoreID = ?", [$store_id]);
        if ($stChk === false) fail('DB error (store scope check)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        $rChk = sqlsrv_fetch_array($stChk, SQLSRV_FETCH_ASSOC);
        if (!$rChk || trim((string)$rChk['Province']) !== trim((string)$scope['province'])) {
            json_response(['data' => [], 'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province']]);
        }
    }

    $where = "WHERE ap.StoreID = ? AND ap.DateKey >= ? AND ap.DateKey < ?";
    $params = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND ap.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND ap.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND ap.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // 1) Top products by total sales in range
    $sqlTop = "
        SELECT TOP ($limit)
            ap.ProductID AS product_id,
            MAX(ap.ProductTitle) AS product_title,
            ap.BrandID AS brand_id,
            MAX(ap.BrandTitle) AS brand_title,
            ap.L1ID AS l1_id,
            CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), ap.SalesAmount))) AS total_sales
        FROM BI.AggDailyProduct ap
        $where
        GROUP BY ap.ProductID, ap.BrandID, ap.L1ID
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), ap.SalesAmount)) DESC
    ";

    $stTop = sqlsrv_query($conn, $sqlTop, $params);
    if ($stTop === false) fail('DB error (top products)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sqlTop]);

    $top = [];
    $productIds = [];
    while ($r = sqlsrv_fetch_array($stTop, SQLSRV_FETCH_ASSOC)) {
        $pid = (int)$r['product_id'];
        $productIds[] = $pid;
        $top[] = [
            'product_id' => $pid,
            'product_title' => (string)($r['product_title'] ?? ''),
            'brand_id' => (int)$r['brand_id'],
            'brand_title' => (string)($r['brand_title'] ?? ''),
            'l1_id' => (int)$r['l1_id'],
            'total_sales' => (string)($r['total_sales'] ?? '0'),
        ];
    }

    if (!$productIds) {
        json_response([
            'data' => [
                'store_id' => $store_id,
                'top_products' => [],
                'months' => [],
                'series' => [],
            ],
            'meta' => [
                'ok' => true,
                'empty' => true,
                'reason' => 'no_data',
                'filters' => [
                    'store_id' => $store_id,
                    'date_from' => $date_from,
                    'date_to' => $date_to,
                ],
                'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
            ]
        ]);
    }

    // 2) Monthly series for those products
    $in = implode(',', array_fill(0, count($productIds), '?'));

    $sqlSeries = "
        SELECT
            CONVERT(char(7), ap.DateKey, 120) AS ym, -- YYYY-MM
            ap.ProductID AS product_id,
            CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), ap.SalesAmount))) AS sales_amount
        FROM BI.AggDailyProduct ap
        $where
          AND ap.ProductID IN ($in)
        GROUP BY CONVERT(char(7), ap.DateKey, 120), ap.ProductID
        ORDER BY ym ASC
    ";

    $paramsSeries = array_merge($params, $productIds);

    $stS = sqlsrv_query($conn, $sqlSeries, $paramsSeries);
    if ($stS === false) fail('DB error (monthly series)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sqlSeries]);

    $monthsSet = [];
    $matrix = []; // matrix[product_id][ym] = sales_amount
    while ($r = sqlsrv_fetch_array($stS, SQLSRV_FETCH_ASSOC)) {
        $ym = (string)$r['ym'];
        $pid = (int)$r['product_id'];
        $monthsSet[$ym] = true;
        if (!isset($matrix[$pid])) $matrix[$pid] = [];
        $matrix[$pid][$ym] = (string)($r['sales_amount'] ?? '0');
    }

    $months = array_keys($monthsSet);
    sort($months);

    $series = [];
    foreach ($top as $p) {
        $pid = $p['product_id'];
        $vals = [];
        foreach ($months as $ym) {
            $vals[] = $matrix[$pid][$ym] ?? '0';
        }
        $series[] = [
            'product_id' => $pid,
            'product_title' => $p['product_title'],
            'brand_title' => $p['brand_title'],
            'l1_id' => $p['l1_id'],
            'values' => $vals,
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'top_products' => $top,
            'months' => $months,
            'series' => $series,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'store_id' => $store_id,
                'date_from' => $date_from,
                'date_to' => $date_to,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id' => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
                'limit' => $limit,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.AggDailyProduct (monthly rollup)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
