<?php
// /api/stores/selected/online_vs_offline_weekly.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') {
        fail('store_id is required', 400);
    }
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');

    // اعمال محدودیت دسترسی از scope
    if (!empty($scope['brand_id'])) {
        $brand_id = (int)$scope['brand_id'];
    }

    $weeks = (int)get_param('weeks', 12);
    if ($weeks <= 0) $weeks = 12;
    if ($weeks > 52) $weeks = 52;

    // فقط n هفته آخر نسبت به date_to را در نظر بگیر
    $window_from = date('Y-m-d', strtotime($date_to . ' -' . ((($weeks * 7) - 1)) . ' days'));
    if ($window_from > $date_from) {
        $date_from = $window_from;
    }

    $conn = db_connect();

    // بررسی محدودهٔ استان برای مدیر استانی
    if (!empty($scope['province'])) {
        $stChk = sqlsrv_query($conn, "SELECT TOP 1 Province FROM BI.DimStore WHERE StoreID = ?", [$store_id]);
        if ($stChk === false) fail('DB error (store scope check)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        $rChk = sqlsrv_fetch_array($stChk, SQLSRV_FETCH_ASSOC);
        if (!$rChk || trim((string)$rChk['Province']) !== trim((string)$scope['province'])) {
            json_response(['data' => [], 'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province']]);
        }
    }

    // فیلترهای اصلی
    $where = "WHERE a.StoreID = ? AND a.DateKey >= ? AND a.DateKey < ?";
    $params = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') {
        $where .= " AND a.BrandID = ?";
        $params[] = (int)$brand_id;
    }

    if ($l1_id !== null && $l1_id !== '') {
        $where .= " AND a.L1ID = ?";
        $params[] = (int)$l1_id;
    }

    // هفته‌بندی بر اساس دوشنبه (مبنای '1900-01-01')
    $sql = "
        SELECT
            CONVERT(date,
                DATEADD(day, -((DATEDIFF(day, '19000101', a.DateKey)) % 7), a.DateKey)
            ) AS week_start,

            SUM(CASE
                    WHEN a.SaleTypeID IN (4,7)
                        THEN TRY_CONVERT(decimal(38,0), a.SalesAmount)
                    ELSE 0
                END) AS offline_sales,

            SUM(CASE
                    WHEN a.SaleTypeID IS NULL
                        THEN TRY_CONVERT(decimal(38,0), a.SalesAmount) -- NULL را آنلاین حساب کن
                    WHEN a.SaleTypeID NOT IN (4,7)
                        THEN TRY_CONVERT(decimal(38,0), a.SalesAmount)
                    ELSE 0
                END) AS online_sales

        FROM BI.AggDailyStoreBrandL1 a
        $where
        GROUP BY
            CONVERT(date,
                DATEADD(day, -((DATEDIFF(day, '19000101', a.DateKey)) % 7), a.DateKey)
            )
        ORDER BY week_start ASC
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) fail('DB error (online_vs_offline_weekly)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $week_start = $r['week_start'];
        $items[] = [
            'week_start'     => is_object($week_start) ? $week_start->format('Y-m-d') : (string)$week_start,
            'offline_sales'  => (string)($r['offline_sales'] ?? '0'),
            'online_sales'   => (string)($r['online_sales'] ?? '0'),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'items'    => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $scope['province'] ?? null,
                'brand_id' => $scope['brand_id'] ?? null,
            ],
            'filters' => [
                'store_id'  => $store_id,
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id'  => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'     => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'weeks'     => $weeks,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.AggDailyStoreBrandL1 (SaleTypeID IN (4,7)=offline, others=online)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
