<?php
// /api/sales/weekday-average.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-29 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // فیلترها (صفحه ۱)
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $city     = trim((string)get_param('city', ''));
    $province = trim((string)get_param('province', ''));
    $area     = trim((string)get_param('area', '')); // فقط تهران
    $sale_type_id = get_param('sale_type_id'); // اختیاری

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id']) $brand_id = $scope['brand_id'];

    $conn = db_connect();

    $joinStore = "";
    $whereExtra = "";
    $paramsExtra = [];

    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $whereExtra .= " AND ds.Province = ?"; $paramsExtra[] = $province; }
        if ($city !== '')     { $whereExtra .= " AND ds.City = ?";     $paramsExtra[] = $city; }
        if ($area !== '')     { $whereExtra .= " AND ds.Area = ?";     $paramsExtra[] = $area; }
    }

    if ($brand_id !== null && $brand_id !== '') { $whereExtra .= " AND a.BrandID = ?"; $paramsExtra[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $whereExtra .= " AND a.L1ID = ?";    $paramsExtra[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $whereExtra .= " AND a.SaleTypeID = ?"; $paramsExtra[] = (int)$sale_type_id; }

    // اول جمع فروش هر روز
    $sqlDaily = "
        SELECT
            a.DateKey AS [date],
            SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS day_sales
        FROM BI.AggDailyStoreBrandL1 a
        $joinStore
        WHERE a.DateKey >= ? AND a.DateKey < ?
        $whereExtra
        GROUP BY a.DateKey
    ";

    $params = array_merge([$date_from, $date_to_excl], $paramsExtra);
    $stmt = sqlsrv_query($conn, $sqlDaily, $params);
    if ($stmt === false) sqlsrv_fail('DB error (weekday daily)', 500);

    // محاسبه میانگین بر اساس روز هفته در PHP (سریع و بی‌دردسر)
    // ترتیب هفته: Sat..Fri
    $map = [
        0 => ['key' => 0, 'label' => 'شنبه'],
        1 => ['key' => 1, 'label' => 'یکشنبه'],
        2 => ['key' => 2, 'label' => 'دوشنبه'],
        3 => ['key' => 3, 'label' => 'سه‌شنبه'],
        4 => ['key' => 4, 'label' => 'چهارشنبه'],
        5 => ['key' => 5, 'label' => 'پنجشنبه'],
        6 => ['key' => 6, 'label' => 'جمعه'],
    ];

    $sum = array_fill(0, 7, '0');   // decimal as string via bc? (اینجا با float نمی‌ریم)
    $cnt = array_fill(0, 7, 0);

    // چون day_sales decimal(38,0) است، آن را string می‌گیریم
    while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $d = (string)$row['date']; // YYYY-MM-DD
        $sales = $row['day_sales'];
        if ($sales === null) $sales = 0;

        // تبدیل روز هفته به شنبه=0 ... جمعه=6
        // PHP: 0=Sunday ... 6=Saturday
        $phpDow = (int)date('w', strtotime($d));
        $satBased = ($phpDow + 1) % 7; // Saturday->0, Sunday->1, ... Friday->6

        // جمع امن با decimal بزرگ: با bcadd اگر موجود بود، وگرنه string+int fallback
        $salesStr = is_string($sales) ? $sales : (string)$sales;

        if (function_exists('bcadd')) {
            $sum[$satBased] = bcadd($sum[$satBased], $salesStr, 0);
        } else {
            // fallback: تا وقتی overflow نزنه (معمولاً روزانه ok)، ولی بهتره bcmath نصب باشه
            $sum[$satBased] = (string)((float)$sum[$satBased] + (float)$salesStr);
        }
        $cnt[$satBased] += 1;
    }

    $out = [];
    for ($i=0; $i<7; $i++) {
        $avg = '0';
        if ($cnt[$i] > 0) {
            if (function_exists('bcdiv')) {
                $avg = bcdiv($sum[$i], (string)$cnt[$i], 0);
            } else {
                $avg = (string)floor(((float)$sum[$i]) / $cnt[$i]);
            }
        }
        $out[] = [
            'weekday' => $map[$i]['key'],
            'label'   => $map[$i]['label'],
            'avg_sales' => $avg, // string
            'days' => $cnt[$i],
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => $out,
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'filters' => [
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
