<?php
// /api/brands/online_vs_inperson.php  (weekday breakdown Sat..Fri)
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

function resolve_range_or_fail(): array {
    $month_from = trim((string)get_param('month_from', ''));
    $month_to   = trim((string)get_param('month_to', ''));

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if ($month_from !== '' && $month_to !== '') {
        if (!preg_match('/^\d{4}\-\d{2}$/', $month_from) || !preg_match('/^\d{4}\-\d{2}$/', $month_to)) {
            fail('Invalid month_from/month_to. Use YYYY-MM', 400);
        }
        $date_from = $month_from . '-01';
        $date_to   = date('Y-m-t', strtotime($month_to . '-01'));
    }

    if (!$date_from || !$date_to) {
        fail('month_from/month_to (or date_from/date_to) are required', 400);
    }

    $date_to_excl = add_days($date_to, 1);
    return [$date_from, $date_to, $date_to_excl];
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    [$date_from, $date_to, $date_to_excl] = resolve_range_or_fail();

    // Filters
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // Role scope
    if (!empty($scope['province'])) $province = (string)$scope['province'];
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    // area فقط تهران
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    $where  = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { 
        $where .= " AND a.BrandID = ?"; 
        $params[] = (int)$brand_id; 
    }
    if ($l1_id !== null && $l1_id !== '') {       
        $where .= " AND a.L1ID = ?";    
        $params[] = (int)$l1_id; 
    }

    $joinStore = "";
    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
    }

    // شنبه تا جمعه
    // DATEFIRST 6 => شنبه=1 ... جمعه=7
    $sql = "
        SET DATEFIRST 6;

        WITH base AS (
            SELECT
                DATEPART(WEEKDAY, CAST(a.DateKey AS date)) AS wd,
                a.SaleTypeID,
                TRY_CONVERT(decimal(38,0), a.SalesAmount) AS amt
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $where
        )
        SELECT
            wd,
            CONVERT(varchar(60),
                SUM(
                    CASE 
                        WHEN SaleTypeID IN (4,7) THEN amt 
                        ELSE 0 
                    END
                )
            ) AS offline_sales,
            CONVERT(varchar(60),
                SUM(
                    CASE 
                        WHEN SaleTypeID NOT IN (4,7) THEN amt 
                        ELSE 0 
                    END
                )
            ) AS online_sales
        FROM base
        GROUP BY wd
        ORDER BY wd
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        fail('DB error (online_vs_inperson weekdays)', 500, [
            'sqlsrv_errors' => sqlsrv_errors(),
            'sql' => $sql
        ]);
    }

    $labels = [
        1 => 'شنبه',
        2 => 'یکشنبه',
        3 => 'دوشنبه',
        4 => 'سه‌شنبه',
        5 => 'چهارشنبه',
        6 => 'پنجشنبه',
        7 => 'جمعه',
    ];

    // پر کردن همه روزها حتی اگر دیتایی نباشد
    $map = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $wd = (int)($r['wd'] ?? 0);
        if ($wd < 1 || $wd > 7) continue;
        $map[$wd] = [
            'weekday' => $labels[$wd],
            'weekday_num' => $wd,
            'offline_sales' => (string)($r['offline_sales'] ?? '0'),
            'online_sales'  => (string)($r['online_sales'] ?? '0'),
        ];
    }

    $items = [];
    for ($wd = 1; $wd <= 7; $wd++) {
        $items[] = $map[$wd] ?? [
            'weekday' => $labels[$wd],
            'weekday_num' => $wd,
            'offline_sales' => '0',
            'online_sales'  => '0',
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $scope['province'] ?? null,
                'brand_id' => $scope['brand_id'] ?? null,
            ],
            'filters' => [
                'month_from' => get_param('month_from') ?: null,
                'month_to'   => get_param('month_to') ?: null,
                'date_from'  => $date_from,
                'date_to'    => $date_to,
                'brand_id'   => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'      => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'province'   => $province ?: null,
                'city'       => $city ?: null,
                'area'       => $area ?: null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'note' => 'Sat..Fri based on DATEFIRST 6',
            'source' => 'BI.AggDailyStoreBrandL1 (SaleTypeID IN (4,7)=offline, others=online)',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
