<?php
// /api/lookup/stores.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';
require_once __DIR__ . '/../../src/helpers/DateRange.php'; // همون helper ماه

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    [$date_from, $date_to, $date_to_excl, $mode] = resolve_date_range();

    $q = trim((string)get_param('q', ''));
    $city = trim((string)get_param('city', ''));
    $area = trim((string)get_param('area', ''));
    $brand_id = get_param('brand_id');

    // role scope
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    // area فقط تهران
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    $limit = (int)get_param('limit', 50);
    if ($limit <= 0) $limit = 50;
    if ($limit > 2000) $limit = 2000;

    $conn = db_connect();

    // store list from AggDailyStoreBrandL1 (data-aware) + join DimStore
    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }

    // Province manager scope
    $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
    if (!empty($scope['province'])) { $where .= " AND ds.Province = ?"; $params[] = $scope['province']; }

    if ($city !== '') { $where .= " AND ds.City = ?"; $params[] = $city; }
    if ($area !== '') { $where .= " AND ds.Area = ?"; $params[] = $area; }

    if ($q !== '') {
        // search by name/code
        $where .= " AND (ds.StoreName LIKE ? OR ds.YaranName LIKE ? OR CAST(ds.StoreID AS nvarchar(30)) = ? OR ds.YaranCode LIKE ?)";
        $params[] = '%' . $q . '%';
        $params[] = '%' . $q . '%';
        $params[] = $q;
        $params[] = '%' . $q . '%';
    }

    $sql = "
        SELECT TOP ($limit)
            ds.StoreID AS store_id,
            MAX(ds.StoreName) AS store_name,
            MAX(ds.City) AS city,
            MAX(ds.Area) AS area,
            MAX(ds.YaranCode) AS yaran_code,
            MAX(ds.YaranName) AS yaran_name,
            MAX(ds.Lat_Long) AS lat_long
        FROM BI.AggDailyStoreBrandL1 a
        $joinStore
        $where
        GROUP BY ds.StoreID
        ORDER BY MAX(ds.StoreName) ASC
    ";

    $st = sqlsrv_query($conn, $sql, $params);
    if ($st === false) fail('DB error (lookup stores)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);

    $items = [];
    while ($r = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'store_id' => (int)$r['store_id'],
            'store_name' => (string)($r['store_name'] ?? ''),
            'city' => (string)($r['city'] ?? ''),
            'area' => (string)($r['area'] ?? ''),
            'yaran_code' => (string)($r['yaran_code'] ?? ''),
            'yaran_name' => (string)($r['yaran_name'] ?? ''),
            'lat_long' => (string)($r['lat_long'] ?? ''),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => $items,
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'q' => $q ?: null,
                'city' => $city ?: null,
                'area' => $area ?: null,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'month_from' => get_param('month_from') ?: null,
                'month_to' => get_param('month_to') ?: null,
                'date_from' => $date_from,
                'date_to' => $date_to,
                'limit' => $limit,
            ],
            'duration_ms' => $elapsed_ms,
            'source' => 'AggDailyStoreBrandL1 + DimStore',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
