<?php
// /api/l1/top5.php  (REAL L1 = 12 groups)
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

$t0 = microtime(true);

$user  = auth_user();
$scope = require_role_scope_filters($user);

$conn = db_connect();
if (!$conn) fail('DB connection not available. (conn is null)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

// ---- L1 MASTER (12 items) ----
$L1_MAP = [
  1  => 'نوشیدنی',
  2  => 'لبنیات',
  3  => 'شوینده و بهداشتی',
  4  => 'کمپوت و کنسرو',
  5  => 'یخچالی و انجمادی',
  6  => 'چاشنی و افزودنی ها',
  7  => 'دخانیات',
  8  => 'خشکبار و تنقلات',
  9  => 'دسر و شیرینی',
  10 => 'چای ، قهوه ، محصولات پودری',
  11 => 'لوازم آشپزخانه ، مصرفی و فرهنگی',
  12 => 'خواروبار و کالای اساسی',
];

// params
$date_from = parse_date(get_param('date_from'));
$date_to   = parse_date(get_param('date_to')); // inclusive

if (!$date_from || !$date_to) {
    $date_to = date('Y-m-d');
    $date_from = date('Y-m-d', strtotime('-89 days'));
}
$date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

$limit = (int)get_param('limit', 5);
if ($limit <= 0 || $limit > 12) $limit = 5;

$brand_id = get_param('brand_id');
$province = trim((string)get_param('province', ''));
$city     = trim((string)get_param('city', ''));
$area     = trim((string)get_param('area', ''));
$store_id = get_param('store_id');
$sale_type_id = get_param('sale_type_id');

// scope
if (!empty($scope['province'])) $province = $scope['province'];
if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

$brand_id = ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null;
$store_id = ($store_id !== null && $store_id !== '') ? (int)$store_id : null;
$sale_type_id = ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null;

// build WHERE
$where  = "WHERE a.DateKey >= ? AND a.DateKey < ?";
$params = [$date_from, $date_to_excl];

if ($brand_id !== null) { $where .= " AND a.BrandID = ?"; $params[] = $brand_id; }
if ($store_id !== null) { $where .= " AND a.StoreID = ?"; $params[] = $store_id; }
if ($sale_type_id !== null) { $where .= " AND a.SaleTypeID = ?"; $params[] = $sale_type_id; }

// city/area/province optional
$joinStore = "";
if ($province !== '' || $city !== '' || $area !== '') {
    $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
    if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
}

// TOTAL (for share%)
$sqlTotal = "
SELECT SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales
FROM BI.AggDailyStoreBrandL1 a
$joinStore
$where
AND a.L1ID BETWEEN 1 AND 12
";
$stT = sqlsrv_query($conn, $sqlTotal, $params);
if ($stT === false) fail('DB error (l1 top5 total)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
$rt = sqlsrv_fetch_array($stT, SQLSRV_FETCH_ASSOC) ?: [];
$total_sales = (string)($rt['total_sales'] ?? '0');
$totalNum = (float)$total_sales;

// TOP L1 (IDs only from 1..12) — titles from L1_MAP
$sql = "
SELECT TOP ($limit)
  a.L1ID AS l1_id,
  SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales
FROM BI.AggDailyStoreBrandL1 a
$joinStore
$where
AND a.L1ID BETWEEN 1 AND 12
GROUP BY a.L1ID
ORDER BY sales DESC
";
$st = sqlsrv_query($conn, $sql, $params);
if ($st === false) fail('DB error (l1 top5)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

$items = [];
while ($r = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC)) {
    $id = (int)$r['l1_id'];
    $salesStr = (string)($r['sales'] ?? '0');
    $salesNum = (float)$salesStr;
    $items[] = [
        'l1_id' => $id,
        'l1_title' => $L1_MAP[$id] ?? ('L1-' . $id),
        'sales' => $salesStr,
        'share_percent' => ($totalNum > 0 ? round(($salesNum / $totalNum) * 100, 2) : 0.0),
    ];
}

json_response([
  'data' => [
      'total_sales' => $total_sales,
      'items' => $items,
  ],
  'meta' => [
      'ok' => true,
      'role' => $user['Role'] ?? null,
      'filters' => [
          'date_from' => $date_from,
          'date_to' => $date_to,
          'limit' => $limit,
          'brand_id' => $brand_id,
          'province' => $province ?: null,
          'city' => $city ?: null,
          'area' => $area ?: null,
          'store_id' => $store_id,
          'sale_type_id' => $sale_type_id,
      ],
      'source' => 'BI.AggDailyStoreBrandL1 (L1ID 1..12) + PHP L1_MAP',
      'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
      'amount_unit' => 'IRR'
  ]
]);
