<?php
// /api/categories/share.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

function q(string $name): string {
    // QUOTENAME safe for identifiers
    return '[' . str_replace(']', ']]', $name) . ']';
}

/**
 * پیدا کردن جدول/ستون‌های L1 از ProductGroup
 * خروجی: ['full'=>'dbo.ProductGroup', 'id_col'=>'Id', 'title_col'=>'ProductGroupName']
 */
function detect_product_group_table($conn): array {
    // Candidate full names (اولویت با BI، بعد dbo)
    $candidates = ['BI.ProductGroup', 'dbo.ProductGroup', 'ProductGroup'];

    $foundFull = null;
    foreach ($candidates as $full) {
        $sql = "SELECT OBJECT_ID(?) AS oid";
        $st = sqlsrv_query($conn, $sql, [$full]);
        if ($st && ($r = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC))) {
            if (!empty($r['oid'])) {
                $foundFull = $full;
                break;
            }
        }
    }
    if (!$foundFull) {
        throw new Exception("ProductGroup table not found (checked: BI.ProductGroup, dbo.ProductGroup, ProductGroup)");
    }

    // ستون‌ها
    $sqlCols = "
        SELECT c.name
        FROM sys.columns c
        WHERE c.object_id = OBJECT_ID(?)
    ";
    $stc = sqlsrv_query($conn, $sqlCols, [$foundFull]);
    if ($stc === false) sqlsrv_fail('DB error (detect productgroup columns)', 500);

    $cols = [];
    while ($r = sqlsrv_fetch_array($stc, SQLSRV_FETCH_ASSOC)) {
        $cols[] = $r['name'];
    }

    // گزینه‌های ممکن برای ID و Title
    $idCandidates = ['Id', 'ID', 'ProductGroupID', 'ProductGroupId', 'GroupID', 'GroupId'];
    $titleCandidates = ['ProductGroupName', 'ProductGroupTitle', 'Title', 'Name', 'GroupName'];

    $idCol = null;
    foreach ($idCandidates as $c) if (in_array($c, $cols, true)) { $idCol = $c; break; }

    $titleCol = null;
    foreach ($titleCandidates as $c) if (in_array($c, $cols, true)) { $titleCol = $c; break; }

    if (!$idCol || !$titleCol) {
        throw new Exception("ProductGroup columns not recognized. Found columns: " . implode(', ', $cols));
    }

    return ['full' => $foundFull, 'id_col' => $idCol, 'title_col' => $titleCol];
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $conn = db_connect();

    // تشخیص جدول و ستون‌های L1
    $pg = detect_product_group_table($conn);
    // $pg['full'], $pg['id_col'], $pg['title_col']

    // تاریخ‌ها: اگر پارامتر نداد → آخرین ماه موجود در Agg
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if (!$date_from || !$date_to) {
        $sqlMax = "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1";
        $stMax = sqlsrv_query($conn, $sqlMax);
        if ($stMax === false) sqlsrv_fail('DB error (max date)', 500);

        $mx = null;
        if ($r = sqlsrv_fetch_array($stMax, SQLSRV_FETCH_ASSOC)) {
            $mx = $r['mx'] ?? null;
        }

        if (!$mx) {
            json_response([
                'data' => ['total_sales' => '0', 'items' => []],
                'meta' => ['ok' => true, 'note' => 'No data in Agg table']
            ]);
            exit;
        }

        $date_from = date('Y-m-01', strtotime($mx));
        $date_to   = date('Y-m-t',  strtotime($mx));
    }

    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // فیلترها (صفحه ۱)
    $brand_id = get_param('brand_id');
    $city     = trim((string)get_param('city', ''));
    $province = trim((string)get_param('province', ''));
    $area     = trim((string)get_param('area', '')); // فقط تهران
    $sale_type_id = get_param('sale_type_id');

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id']) $brand_id = $scope['brand_id'];

    $joinStore = "";
    $whereExtra = "";
    $paramsExtra = [];

    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $whereExtra .= " AND ds.Province = ?"; $paramsExtra[] = $province; }
        if ($city !== '')     { $whereExtra .= " AND ds.City = ?";     $paramsExtra[] = $city; }
        if ($area !== '')     { $whereExtra .= " AND ds.Area = ?";     $paramsExtra[] = $area; }
    }

    if ($brand_id !== null && $brand_id !== '') { $whereExtra .= " AND a.BrandID = ?"; $paramsExtra[] = (int)$brand_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $whereExtra .= " AND a.SaleTypeID = ?"; $paramsExtra[] = (int)$sale_type_id; }

    // join به ProductGroup با نام/ستون‌های واقعی
    $pgFull = $pg['full']; // مثل dbo.ProductGroup یا BI.ProductGroup
    $pgId   = q($pg['id_col']);
    $pgTit  = q($pg['title_col']);

    // اگر full مثل "dbo.ProductGroup" بود باید با [] safe کنیم
    $parts = explode('.', $pgFull);
    if (count($parts) == 2) {
        $pgTable = q($parts[0]) . "." . q($parts[1]);
    } else {
        $pgTable = q('dbo') . "." . q($pgFull); // fallback
    }

  $sql = "
    SELECT
        a.L1ID AS l1_id,
        ISNULL(l1.L1Title, CONCAT('L1 #', a.L1ID)) AS l1_title,
        CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS varchar(50)) AS sales
    FROM BI.AggDailyStoreBrandL1 a
    LEFT JOIN BI.DimL1 l1 ON l1.L1ID = a.L1ID
    $joinStore
    WHERE a.DateKey >= ? AND a.DateKey < ?
    $whereExtra
    GROUP BY a.L1ID, l1.L1Title
    ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC
";


    $params = array_merge([$date_from, $date_to_excl], $paramsExtra);
    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail('DB error (categories share)', 500);

    $rows = [];
    $total = '0';

    while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $sales = (string)($row['sales'] ?? '0');

        if (function_exists('bcadd')) $total = bcadd($total, $sales, 0);
        else $total = (string)((float)$total + (float)$sales);

        $rows[] = [
            'l1_id' => (int)$row['l1_id'],
            'l1_title' => (string)$row['l1_title'],
            'sales' => $sales,
        ];
    }

    $items = [];
    foreach ($rows as $r) {
        $pct = 0.0;
        if ($total !== '0') $pct = 0.0;
if ($total !== '0') {
    $pct = ((float)$r['sales'] / (float)$total) * 100.0;
}

$pct_raw = round($pct, 4); // درصد واقعی با دقت بیشتر
$pct_display = round($pct, 2); // برای نمایش

// اگر صفر شد ولی واقعاً سهم داشته، حداقل 0.01 نشون بده
if ($pct_display == 0.0 && (float)$r['sales'] > 0) {
    $pct_display = 0.01;
}

$items[] = [
    'l1_id' => $r['l1_id'],
    'l1_title' => $r['l1_title'],
    'sales' => $r['sales'],
    'share_percent' => $pct_display,      // برای UI
    'share_percent_raw' => $pct_raw,      // برای محاسبات دقیق/tooltip
];

    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'total_sales' => $total,
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'l1_source' => [
                'table' => $pgFull,
                'id_col' => $pg['id_col'],
                'title_col' => $pg['title_col'],
            ],
            'filters' => [
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
