<?php
// /api/brands/drilldown.php  (Brand Drilldown -> TOP Products from BI.SaleDetail)
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $conn = db_connect();

    // required
    $brand_id = get_param('brand_id');
    if ($brand_id === null || $brand_id === '') fail("Missing brand_id", 400);
    $brand_id = (int)$brand_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) fail("Missing/invalid date_from/date_to", 400);
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // optional filters
    $l1_id = get_param('l1_id'); // چون مپینگ L1 در دیتای شما قاطیه، اینجا مقاوم می‌زنیم
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));
    $store_id = get_param('store_id');

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id'] && (int)$scope['brand_id'] !== $brand_id) {
        fail("Forbidden (brand scope)", 403);
    }

    $limit = (int)get_param('limit', 10);
    if ($limit <= 0 || $limit > 50) $limit = 10;

    // WHERE
    $where = "WHERE sd.FactorDate >= ? AND sd.FactorDate < ? AND sd.BrandID = ?";
    $params = [$date_from, $date_to_excl, $brand_id];

    // لایه مقاومت برای L1 (هرکدوم از این دو ستون ممکنه L1 شما باشه)
    if ($l1_id !== null && $l1_id !== '') {
        $where .= " AND (sd.ProductGroupID = ? OR sd.ProductCategoryID = ?)";
        $params[] = (int)$l1_id;
        $params[] = (int)$l1_id;
    }

    if ($sale_type_id !== null && $sale_type_id !== '') {
        $where .= " AND sd.SaleTypeID = ?";
        $params[] = (int)$sale_type_id;
    }

    if ($province !== '') { $where .= " AND sd.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND sd.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND sd.Area = ?";     $params[] = $area; }
    if ($store_id !== null && $store_id !== '') { $where .= " AND sd.StoreID = ?"; $params[] = (int)$store_id; }

    // TOP Products (قطعی‌ترین Drilldown)
    $sql = "
        SELECT TOP ($limit)
            sd.ProductID AS product_id,
            ISNULL(NULLIF(LTRIM(RTRIM(sd.ProductTitle)),''), N'نامشخص') AS product_title,
            CAST(SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS varchar(50)) AS sales,
            CAST(SUM(TRY_CONVERT(decimal(38,3), sd.Quantity)) AS varchar(50)) AS qty
        FROM BI.SaleDetail sd
        $where
        GROUP BY sd.ProductID, sd.ProductTitle
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail('DB error (brand drilldown products)', 500);

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'product_id' => (int)$r['product_id'],
            'product_title' => (string)$r['product_title'],
            'sales' => (string)($r['sales'] ?? '0'),
            'qty' => (string)($r['qty'] ?? '0'),
        ];
    }

    // اگر به خاطر L1 خالی شد، یک بار بدون L1 برگردون تا حداقل خروجی قطعی داشته باشیم
    $fallback_used = false;
    if (count($items) === 0 && $l1_id !== null && $l1_id !== '') {
        $fallback_used = true;

        $where2 = "WHERE sd.FactorDate >= ? AND sd.FactorDate < ? AND sd.BrandID = ?";
        $params2 = [$date_from, $date_to_excl, $brand_id];

        if ($sale_type_id !== null && $sale_type_id !== '') { $where2 .= " AND sd.SaleTypeID = ?"; $params2[] = (int)$sale_type_id; }
        if ($province !== '') { $where2 .= " AND sd.Province = ?"; $params2[] = $province; }
        if ($city !== '')     { $where2 .= " AND sd.City = ?";     $params2[] = $city; }
        if ($area !== '')     { $where2 .= " AND sd.Area = ?";     $params2[] = $area; }
        if ($store_id !== null && $store_id !== '') { $where2 .= " AND sd.StoreID = ?"; $params2[] = (int)$store_id; }

        $sql2 = "
            SELECT TOP ($limit)
                sd.ProductID AS product_id,
                ISNULL(NULLIF(LTRIM(RTRIM(sd.ProductTitle)),''), N'نامشخص') AS product_title,
                CAST(SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS varchar(50)) AS sales,
                CAST(SUM(TRY_CONVERT(decimal(38,3), sd.Quantity)) AS varchar(50)) AS qty
            FROM BI.SaleDetail sd
            $where2
            GROUP BY sd.ProductID, sd.ProductTitle
            ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
            OPTION (RECOMPILE);
        ";

        $stmt2 = sqlsrv_query($conn, $sql2, $params2);
        if ($stmt2 === false) sqlsrv_fail('DB error (brand drilldown fallback)', 500);

        $items = [];
        while ($r = sqlsrv_fetch_array($stmt2, SQLSRV_FETCH_ASSOC)) {
            $items[] = [
                'product_id' => (int)$r['product_id'],
                'product_title' => (string)$r['product_title'],
                'sales' => (string)($r['sales'] ?? '0'),
                'qty' => (string)($r['qty'] ?? '0'),
            ];
        }
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'brand_id' => $brand_id,
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'source' => 'BI.SaleDetail',
            'fallback_used' => $fallback_used,
            'filters' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
                'brand_id' => $brand_id,
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'limit' => $limit,
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'store_id' => ($store_id === '' ? null : $store_id),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
