<?php
// /api/brands/daily_share.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

function date_add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // ---- optional filters ----
    $l1_id = get_param('l1_id'); // اگر روی AggDailyStoreBrandL1 دارید، این همون L1ID هست
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));
    $store_id = get_param('store_id');

    // role scope
    if ($scope['province']) $province = $scope['province'];

    // ---- build common join/where (بدون تاریخ) ----
    $joinStore = "";
    $whereCommon = "WHERE 1=1";
    $paramsCommon = [];

    if ($province !== '' || $city !== '' || $area !== '' || ($store_id !== null && $store_id !== '')) {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $whereCommon .= " AND ds.Province = ?"; $paramsCommon[] = $province; }
        if ($city !== '')     { $whereCommon .= " AND ds.City = ?";     $paramsCommon[] = $city; }
        if ($area !== '')     { $whereCommon .= " AND ds.Area = ?";     $paramsCommon[] = $area; }
        if ($store_id !== null && $store_id !== '') { $whereCommon .= " AND ds.StoreID = ?"; $paramsCommon[] = (int)$store_id; }
    }

    if ($l1_id !== null && $l1_id !== '') {
        $whereCommon .= " AND a.L1ID = ?";
        $paramsCommon[] = (int)$l1_id;
    }

    if ($sale_type_id !== null && $sale_type_id !== '') {
        $whereCommon .= " AND a.SaleTypeID = ?";
        $paramsCommon[] = (int)$sale_type_id;
    }

    // اگر مدیر برند باشه: همینجا محدودش کن (هرچند نمودار ۷ صفحه ۱ معمولاً برای ادمین/استانه)
    if ($scope['brand_id']) {
        $whereCommon .= " AND a.BrandID = ?";
        $paramsCommon[] = (int)$scope['brand_id'];
    }

    // ---- dates ----
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // اگر تاریخ‌ها داده نشده: آخرین ۱۴ روز از روی MAX(DateKey)
    if (!$date_to) {
        $sqlMax = "
            SELECT MAX(a.DateKey) AS mx
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $whereCommon
        ";
        $stmtMax = sqlsrv_query($conn, $sqlMax, $paramsCommon);
        if ($stmtMax === false) sqlsrv_fail('DB error (max date)', 500);
        $rowMax = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC);
        $mx = $rowMax['mx'] ?? null;
        if (!$mx) fail("No data to determine date range", 400);
        $date_to = (string)$mx;
    }

    if (!$date_from) {
        // ۱۴ روز شامل date_to => 13 روز قبل
        $date_from = date_add_days($date_to, -13);
    }

    $date_to_excl = date_add_days($date_to, 1);

    // ---- 1) find Top 3 brands in range ----
    $whereTop = $whereCommon . " AND a.DateKey >= ? AND a.DateKey < ?";
    $paramsTop = array_merge($paramsCommon, [$date_from, $date_to_excl]);

    $sqlTop3 = "
        SELECT TOP (3)
            a.BrandID AS brand_id,
            ISNULL(db.BrandTitle, CONCAT(N'Brand #', a.BrandID)) AS brand_title,
            SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS s
        FROM BI.AggDailyStoreBrandL1 a
        LEFT JOIN BI.DimBrand db ON db.BrandID = a.BrandID
        $joinStore
        $whereTop
        GROUP BY a.BrandID, db.BrandTitle
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC
    ";
    $stmtTop = sqlsrv_query($conn, $sqlTop3, $paramsTop);
    if ($stmtTop === false) sqlsrv_fail('DB error (top3 brands)', 500);

    $brands = [];
    $brandIds = [];
    while ($r = sqlsrv_fetch_array($stmtTop, SQLSRV_FETCH_ASSOC)) {
        $bid = (int)$r['brand_id'];
        $brandIds[] = $bid;
        $brands[] = [
            'brand_id' => $bid,
            'brand_title' => (string)$r['brand_title'],
        ];
    }

    if (count($brandIds) === 0) {
        json_response([
            'data' => ['brands' => [], 'days' => []],
            'meta' => [
                'ok' => true,
                'note' => 'No brands found in range',
                'filters' => [
                    'date_from' => $date_from,
                    'date_to' => $date_to,
                ]
            ]
        ]);
        exit;
    }

    // ---- 2) daily share for those top 3 vs total daily sales ----
    $in = implode(',', array_fill(0, count($brandIds), '?'));

    $whereRange = $whereCommon . " AND a.DateKey >= ? AND a.DateKey < ?";
    $paramsRange = array_merge($paramsCommon, [$date_from, $date_to_excl]);

    $sql = "
        WITH totals AS (
            SELECT
                a.DateKey,
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $whereRange
            GROUP BY a.DateKey
        ),
        brands AS (
            SELECT
                a.DateKey,
                a.BrandID,
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS brand_sales
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $whereRange
              AND a.BrandID IN ($in)
            GROUP BY a.DateKey, a.BrandID
        )
        SELECT
            b.DateKey AS [date],
            b.BrandID AS brand_id,
            CAST(b.brand_sales AS varchar(50)) AS sales,
            CAST(t.total_sales AS varchar(50)) AS total_sales,
            CAST(
                (CAST(b.brand_sales AS decimal(38,6)) * 100)
                / NULLIF(CAST(t.total_sales AS decimal(38,6)), 0)
            AS decimal(9,2)) AS share_percent
        FROM brands b
        JOIN totals t ON t.DateKey = b.DateKey
        ORDER BY b.DateKey ASC;
    ";

    // چون $whereRange دو بار داخل $sql تکرار شده (totals و brands)
// باید پارامترها هم دو بار ارسال بشن
$params = array_merge($paramsRange, $paramsRange, $brandIds);

$stmt = sqlsrv_query($conn, $sql, $params);

    if ($stmt === false) sqlsrv_fail('DB error (daily share)', 500);

    // build date -> items
    $byDate = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $d = (string)$r['date'];
        if (!isset($byDate[$d])) {
            $byDate[$d] = [
                'date' => $d,
                'total_sales' => (string)$r['total_sales'],
                'items' => []
            ];
        }
        $byDate[$d]['items'][] = [
            'brand_id' => (int)$r['brand_id'],
            'sales' => (string)$r['sales'],
            'share_percent' => (float)$r['share_percent'],
        ];
    }

    // make full 14-day list (حتی اگر روزی فروش 0 بود)
    $days = [];
    $cur = $date_from;
    while ($cur <= $date_to) {
        $days[] = $byDate[$cur] ?? [
            'date' => $cur,
            'total_sales' => "0",
            'items' => []
        ];
        $cur = date_add_days($cur, 1);
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'brands' => $brands,
            'days' => $days
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'range' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
                'days' => 14
            ],
            'filters' => [
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'store_id' => ($store_id === '' ? null : $store_id),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
                'brand_scope' => $scope['brand_id'] ?? null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR'
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
