<?php
// /api/areas/selected_area_summary.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') {
        fail('store_id is required', 400);
    }
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // default: last 90 days
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope (brand manager)
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $conn = db_connect();

    // 1) find selected store area/city/province
    $sqlStore = "
        SELECT TOP 1
            LTRIM(RTRIM(ISNULL(Province, N''))) AS Province,
            LTRIM(RTRIM(ISNULL(City, N'')))     AS City,
            LTRIM(RTRIM(CAST(ISNULL(Area, N'') AS nvarchar(50)))) AS Area
        FROM BI.DimStore
        WHERE StoreID = ?
    ";
    $stS = sqlsrv_query($conn, $sqlStore, [$store_id]);
    if ($stS === false) fail('DB error (area store lookup)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $ds = sqlsrv_fetch_array($stS, SQLSRV_FETCH_ASSOC);

    if (!$ds) {
        json_response([
            'data' => null,
            'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_not_found_in_dimstore'],
        ]);
    }

    $province = $ds['Province'] ?? '';
    $city     = $ds['City'] ?? '';
    $area     = $ds['Area'] ?? '';

    // province manager scope: اگر استان کاربر مشخص است، باید با استان فروشگاه یکی باشد
    if (!empty($scope['province']) && $scope['province'] !== $province) {
        json_response([
            'data' => null,
            'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province'],
        ]);
    }

    // 2) compute area totals from AggDailyStoreBrandL1 joined with DimStore
    $where = "WHERE a.DateKey >= ? AND a.DateKey < ? AND LTRIM(RTRIM(ds.City)) = ? AND LTRIM(RTRIM(CAST(ds.Area AS nvarchar(50)))) = ?";
    $params = [$date_from, $date_to_excl, $city, $area];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND a.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    $sqlTotal = "
        SELECT
            SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales,
            COUNT(DISTINCT a.DateKey) AS days_count
        FROM BI.AggDailyStoreBrandL1 a
        JOIN BI.DimStore ds ON ds.StoreID = a.StoreID
        $where
    ";
    $stT = sqlsrv_query($conn, $sqlTotal, $params);
    if ($stT === false) fail('DB error (area total)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rt = sqlsrv_fetch_array($stT, SQLSRV_FETCH_ASSOC) ?: [];

    $total_sales = (string)($rt['total_sales'] ?? '0');
    $days_count  = (int)($rt['days_count'] ?? 0);

    $avg_daily_sales = '0';
    if ($days_count > 0) {
        $ts = preg_replace('/[^0-9\-]/', '', $total_sales);
        if ($ts === '' || $ts === '-') $ts = '0';
        $avg_daily_sales = (string)intval(((float)$ts) / $days_count);
    }

    // 3) last 7 days sales for this area
    $last7_from = date('Y-m-d', strtotime($date_to . ' -6 days'));
    $where7 = "WHERE a.DateKey >= ? AND a.DateKey < ? AND LTRIM(RTRIM(ds.City)) = ? AND LTRIM(RTRIM(CAST(ds.Area AS nvarchar(50)))) = ?";
    $p7 = [$last7_from, $date_to_excl, $city, $area];

    if ($brand_id !== null && $brand_id !== '') { $where7 .= " AND a.BrandID = ?"; $p7[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where7 .= " AND a.L1ID = ?";    $p7[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where7 .= " AND a.SaleTypeID = ?"; $p7[] = (int)$sale_type_id; }

    $sql7 = "
        SELECT SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales_last_7d
        FROM BI.AggDailyStoreBrandL1 a
        JOIN BI.DimStore ds ON ds.StoreID = a.StoreID
        $where7
    ";
    $st7 = sqlsrv_query($conn, $sql7, $p7);
    if ($st7 === false) fail('DB error (area last7d)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $r7 = sqlsrv_fetch_array($st7, SQLSRV_FETCH_ASSOC) ?: [];
    $sales_last_7d = (string)($r7['sales_last_7d'] ?? '0');

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'province' => $province ?: null,
            'city'     => $city ?: null,
            'area'     => $area ?: null,
            'total_sales_area' => $total_sales,
            'avg_daily_sales_area' => $avg_daily_sales,
            'sales_last_7d_area' => $sales_last_7d,
            'days_count' => $days_count,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'store_id' => $store_id,
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id'  => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'     => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'AggDailyStoreBrandL1 + DimStore',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
