<?php
/**
 * /api/ai/insight/dashboard.php
 * - Reads context from /api/ai/context/dashboard.php
 * - Builds smart narrative + cards
 * - Extracts LAST 14 DAYS from series + detects spikes/outliers
 * - Uses GapGPT if configured (src/config/ai.php + src/helpers/GapGPT.php)
 */

header('Content-Type: application/json; charset=utf-8');

// -----------------------------
// Boot (FIXED): load config + helper directly
// -----------------------------
$ROOT = dirname(__DIR__, 3); // .../dashboard-project

$aiConfigPath = $ROOT . '/src/config/ai.php';
$gapgptPath   = $ROOT . '/src/helpers/GapGPT.php';

$boot_ok = true;
$boot_err = null;

if (file_exists($aiConfigPath)) {
    require_once $aiConfigPath;
} else {
    $boot_ok = false;
    $boot_err = "ai config not found: {$aiConfigPath}";
}

if (file_exists($gapgptPath)) {
    require_once $gapgptPath;
} else {
    $boot_ok = false;
    $boot_err = "GapGPT helper not found: {$gapgptPath}";
}

if ($boot_ok && !function_exists('ai_config')) {
    $boot_ok = false;
    $boot_err = "ai_config() not found after require ai.php";
}
if ($boot_ok && !function_exists('gapgpt_chat')) {
    $boot_ok = false;
    $boot_err = "gapgpt_chat() not found after require GapGPT.php";
}
if ($boot_ok && !function_exists('extract_json_from_text')) {
    $boot_ok = false;
    $boot_err = "extract_json_from_text() not found after require GapGPT.php";
}

// -----------------------------
// Helpers
// -----------------------------
function respond($data, $meta = [], $http = 200) {
    http_response_code($http);
    echo json_encode([
        "data" => $data,
        "meta" => array_merge(["ok" => ($http < 400)], $meta),
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function fail($msg, $http = 400, $debug = null) {
    respond(null, ["ok" => false, "error" => $msg, "debug" => $debug], $http);
}

function safe_float($v, $default = 0.0) {
    if ($v === null) return $default;
    if (is_string($v)) $v = str_replace([',', ' '], '', $v);
    return is_numeric($v) ? (float)$v : $default;
}
function safe_int($v, $default = 0) {
    return (int) round(safe_float($v, $default));
}
function pct($cur, $prev) {
    $cur = safe_float($cur, 0);
    $prev = safe_float($prev, 0);
    if ($prev == 0) return null;
    return (($cur - $prev) / $prev) * 100.0;
}
function irr_to_toman($irr) {
    return (int) floor(safe_float($irr, 0) / 10.0);
}
function fmt_toman($toman) {
    return number_format((float)$toman, 0, '.', ',') . " تومان";
}
function fmt_pct($p) {
    if ($p === null) return "نامشخص";
    $sign = ($p > 0) ? "+" : "";
    return $sign . number_format($p, 2) . "%";
}
function median(array $arr) {
    $arr = array_values(array_filter($arr, fn($x)=>is_numeric($x)));
    $n = count($arr);
    if ($n === 0) return null;
    sort($arr);
    $mid = intdiv($n, 2);
    if ($n % 2) return $arr[$mid];
    return ($arr[$mid - 1] + $arr[$mid]) / 2.0;
}
function mad(array $arr, $med = null) {
    $arr = array_values(array_filter($arr, fn($x)=>is_numeric($x)));
    if (!$arr) return null;
    if ($med === null) $med = median($arr);
    if ($med === null) return null;
    $dev = array_map(fn($x)=>abs($x - $med), $arr);
    return median($dev);
}
function robust_z($x, $med, $mad) {
    if ($med === null || $mad === null || $mad == 0) return null;
    return 0.6745 * (($x - $med) / $mad);
}

function http_get_json($url, $timeout = 25) {
    $ctx = stream_context_create([
        'http' => [
            'method'  => 'GET',
            'timeout' => $timeout,
            'header'  => "Accept: application/json\r\n"
        ]
    ]);
    $raw = @file_get_contents($url, false, $ctx);
    if ($raw === false) return [false, null, null];
    $json = json_decode($raw, true);
    if (!is_array($json)) return [false, $raw, null];
    return [true, $raw, $json];
}

function is_valid_date($s) {
    if (!$s) return false;
    $d = DateTime::createFromFormat('Y-m-d', $s);
    return $d && $d->format('Y-m-d') === $s;
}

function find_by_id($items, $key, $id) {
    foreach ($items as $it) {
        if (isset($it[$key]) && (string)$it[$key] === (string)$id) return $it;
    }
    return null;
}

// -----------------------------
// Inputs
// -----------------------------
$date_from = $_GET['date_from'] ?? null;
$date_to   = $_GET['date_to']   ?? null;

if (!is_valid_date($date_from) || !is_valid_date($date_to)) {
    fail("پارامترهای date_from و date_to باید به فرمت YYYY-MM-DD باشند.");
}

// passthrough filters (optional)
$pass = [
    'brand_id'     => $_GET['brand_id']     ?? null,
    'store_id'     => $_GET['store_id']     ?? null,
    'city'         => $_GET['city']         ?? null,
    'area'         => $_GET['area']         ?? null,
    'l1_id'        => $_GET['l1_id']        ?? null,
    'sale_type_id' => $_GET['sale_type_id'] ?? null,
    'province'     => $_GET['province']     ?? null,
];

// -----------------------------
// Build context URL
// -----------------------------
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host   = $_SERVER['HTTP_HOST'] ?? 'localhost';

$context_url = $scheme . '://' . $host . "/dashboard-project/api/ai/context/dashboard.php"
    . "?date_from=" . urlencode($date_from)
    . "&date_to=" . urlencode($date_to);

foreach ($pass as $k => $v) {
    if ($v !== null && $v !== '') $context_url .= "&{$k}=" . urlencode((string)$v);
}

list($ok_ctx, $raw_ctx, $ctx_json) = http_get_json($context_url, 30);
if (!$ok_ctx || !is_array($ctx_json) || empty($ctx_json['data'])) {
    fail("خواندن context ناموفق بود.", 500, [
        "context_url" => $context_url,
        "context_raw" => is_string($raw_ctx) ? mb_substr($raw_ctx, 0, 600) : null,
    ]);
}

$ctx = $ctx_json['data'];
$kpi = $ctx['kpi'] ?? [];
$overview = $ctx['overview'] ?? [];
$compare = $ctx['compare'] ?? [];
$kpi_prev = $compare['kpi_prev'] ?? [];

// -----------------------------
// Extract KPIs + deltas
// -----------------------------
$total_sales_irr = safe_float($kpi['total_sales'] ?? 0, 0);
$avg_sales_per_store_irr = safe_float($kpi['avg_sales_per_store'] ?? 0, 0);
$last_week_sales_irr = safe_float($kpi['last_week_sales'] ?? 0, 0);
$stores_count = safe_int($kpi['stores_count'] ?? 0, 0);

$total_sales_prev_irr = safe_float($kpi_prev['total_sales'] ?? 0, 0);
$avg_sales_prev_irr   = safe_float($kpi_prev['avg_sales_per_store'] ?? 0, 0);
$last_week_prev_irr   = safe_float($kpi_prev['last_week_sales'] ?? 0, 0);
$stores_prev          = safe_int($kpi_prev['stores_count'] ?? 0, 0);

$total_sales_pct = $compare['kpi_delta']['total_sales_pct'] ?? pct($total_sales_irr, $total_sales_prev_irr);
$avg_sales_pct   = $compare['kpi_delta']['avg_sales_per_store_pct'] ?? pct($avg_sales_per_store_irr, $avg_sales_prev_irr);
$last_week_pct   = $compare['kpi_delta']['last_week_sales_pct'] ?? pct($last_week_sales_irr, $last_week_prev_irr);

$stores_delta = $stores_count - $stores_prev;
$stores_delta_pct = ($stores_prev > 0) ? ($stores_delta / $stores_prev) * 100.0 : null;

$total_toman = irr_to_toman($total_sales_irr);
$avg_toman   = irr_to_toman($avg_sales_per_store_irr);
$last_week_toman = irr_to_toman($last_week_sales_irr);

// -----------------------------
// Overview pieces
// -----------------------------
$top_brands = $overview['top_brands_month']['items'] ?? [];
$top_brands_prev = $compare['top_brands_prev']['items'] ?? [];

$l1_share = $overview['l1_share']['items'] ?? [];
$l1_share_prev = $compare['l1_share_prev']['items'] ?? [];

$top_stores = $overview['top_stores']['items'] ?? [];
$top_stores_prev = $compare['top_stores_prev']['items'] ?? [];

$leader_brand = $top_brands[0] ?? null;
$leader_brand_pp_change = null;
if ($leader_brand && isset($leader_brand['brand_id'])) {
    $prev = find_by_id($top_brands_prev, 'brand_id', $leader_brand['brand_id']);
    if ($prev) $leader_brand_pp_change = safe_float($leader_brand['share_percent'] ?? 0) - safe_float($prev['share_percent'] ?? 0);
}

$leader_l1 = $l1_share[0] ?? null;
$leader_l1_pp_change = null;
if ($leader_l1 && isset($leader_l1['l1_id'])) {
    $prev = find_by_id($l1_share_prev, 'l1_id', $leader_l1['l1_id']);
    if ($prev) $leader_l1_pp_change = safe_float($leader_l1['share_percent'] ?? 0) - safe_float($prev['share_percent'] ?? 0);
}

$leader_store = $top_stores[0] ?? null;
$leader_store_pct_change = null;
if ($leader_store && isset($leader_store['store_id'])) {
    $prev = find_by_id($top_stores_prev, 'store_id', $leader_store['store_id']);
    if ($prev) $leader_store_pct_change = pct($leader_store['sales'] ?? 0, $prev['sales'] ?? 0);
}

$top3_share = 0.0; $top6_share = 0.0;
for ($i=0;$i<min(3,count($top_brands));$i++) $top3_share += safe_float($top_brands[$i]['share_percent'] ?? 0, 0);
for ($i=0;$i<min(6,count($top_brands));$i++) $top6_share += safe_float($top_brands[$i]['share_percent'] ?? 0, 0);

// -----------------------------
// LAST 14 DAYS extraction + spikes/outliers (from series possibly containing full range)
// -----------------------------
$series = $overview['brand_share_14d']['days'] ?? [];
usort($series, fn($a,$b)=>strcmp($a['date'] ?? '', $b['date'] ?? ''));

// unique dates
$seen = [];
$series_u = [];
foreach ($series as $row) {
    $d = $row['date'] ?? null;
    if (!$d) continue;
    if (isset($seen[$d])) continue;
    $seen[$d] = true;
    $series_u[] = $row;
}
$last14 = array_slice($series_u, max(0, count($series_u) - 14), 14);

$last14_totals = array_map(fn($x)=>safe_float($x['total_sales'] ?? 0, 0), $last14);
$med = median($last14_totals);
$madv = mad($last14_totals, $med);

$jump_threshold_pct = 60; // جهش روزانه
$z_threshold = 3.5;       // outlier robust

$spikes = [];
$outliers = [];

for ($i=0; $i<count($last14); $i++) {
    $d = $last14[$i]['date'] ?? '';
    $val = safe_float($last14[$i]['total_sales'] ?? 0, 0);

    $z = robust_z($val, $med, $madv);
    if ($z !== null && abs($z) >= $z_threshold) {
        $outliers[] = ["date"=>$d, "total_sales"=>$val, "z"=>$z];
    }

    if ($i > 0) {
        $prev = safe_float($last14[$i-1]['total_sales'] ?? 0, 0);
        $p = pct($val, $prev);
        if ($p !== null && abs($p) >= $jump_threshold_pct) {
            $spikes[] = ["date"=>$d, "pct"=>$p, "total_sales"=>$val, "prev"=>$prev];
        }
    }
}
usort($spikes, fn($a,$b)=>abs($b['pct']) <=> abs($a['pct']));
usort($outliers, fn($a,$b)=>abs($b['z']) <=> abs($a['z']));

$first7 = array_slice($last14_totals, 0, min(7, count($last14_totals)));
$last7  = array_slice($last14_totals, max(0, count($last14_totals)-7), 7);

$avg_first7 = $first7 ? array_sum($first7)/count($first7) : null;
$avg_last7  = $last7 ? array_sum($last7)/count($last7) : null;
$last7_vs_first7_pct = ($avg_first7 && $avg_first7 != 0) ? (($avg_last7 - $avg_first7)/$avg_first7)*100.0 : null;

$mean14 = (count($last14_totals) ? array_sum($last14_totals)/count($last14_totals) : null);
$std14 = null;
if ($mean14 !== null && count($last14_totals) >= 2) {
    $var = 0.0;
    foreach ($last14_totals as $v) $var += pow($v - $mean14, 2);
    $var /= (count($last14_totals) - 1);
    $std14 = sqrt($var);
}
$cv14 = ($mean14 && $mean14 != 0 && $std14 !== null) ? ($std14 / $mean14) : null;

// -----------------------------
// Build "features_used" for LLM + fallback
// -----------------------------
$features_used = [
    "range" => ["date_from"=>$date_from, "date_to"=>$date_to],
    "kpi" => [
        "total_sales_irr" => (int)$total_sales_irr,
        "avg_sales_per_store_irr" => (int)$avg_sales_per_store_irr,
        "last_week_sales_irr" => (int)$last_week_sales_irr,
        "stores_count" => $stores_count,
    ],
    "compare_prev" => [
        "date_from" => $compare['prev_range']['date_from'] ?? null,
        "date_to"   => $compare['prev_range']['date_to'] ?? null,
        "total_sales_pct" => $total_sales_pct,
        "avg_sales_per_store_pct" => $avg_sales_pct,
        "last_week_sales_pct" => $last_week_pct,
        "stores_delta" => $stores_delta,
        "stores_delta_pct" => $stores_delta_pct,
    ],
    "concentration" => [
        "top3_share_percent" => $top3_share,
        "top6_share_percent" => $top6_share,
    ],
    "leader_brand" => $leader_brand ? [
        "brand_id" => $leader_brand['brand_id'] ?? null,
        "brand_title" => $leader_brand['brand_title'] ?? null,
        "sales_irr" => (int)safe_float($leader_brand['sales'] ?? 0, 0),
        "share_percent" => safe_float($leader_brand['share_percent'] ?? 0, 0),
        "share_pp_change" => $leader_brand_pp_change,
    ] : null,
    "leader_l1" => $leader_l1 ? [
        "l1_id" => $leader_l1['l1_id'] ?? null,
        "l1_title" => $leader_l1['l1_title'] ?? null,
        "sales_irr" => (int)safe_float($leader_l1['sales'] ?? 0, 0),
        "share_percent" => safe_float($leader_l1['share_percent'] ?? 0, 0),
        "share_pp_change" => $leader_l1_pp_change,
    ] : null,
    "leader_store" => $leader_store ? [
        "store_id" => $leader_store['store_id'] ?? null,
        "store_name" => $leader_store['store_name'] ?? null,
        "sales_irr" => (int)safe_float($leader_store['sales'] ?? 0, 0),
        "sales_pct_change" => $leader_store_pct_change,
    ] : null,
    "short_term_last14" => [
        "days" => array_map(fn($x)=>[
            "date"=>$x['date'] ?? null,
            "total_sales_irr" => (int)safe_float($x['total_sales'] ?? 0, 0),
        ], $last14),
        "daily_total_sales_cv" => $cv14,
        "last7_vs_first7_pct" => $last7_vs_first7_pct,
        "spikes" => array_slice($spikes, 0, 5),
        "outliers" => array_slice($outliers, 0, 5),
    ]
];

// -----------------------------
// Fallback builder (no LLM or LLM fail)
// -----------------------------
function build_fallback(array $features, array $ctx) {
    $k = $features['kpi'];
    $c = $features['compare_prev'];
    $lb = $features['leader_brand'];
    $ll1 = $features['leader_l1'];
    $ls = $features['leader_store'];
    $st = $features['short_term_last14'];

    $total_toman = irr_to_toman($k['total_sales_irr']);
    $avg_toman = irr_to_toman($k['avg_sales_per_store_irr']);
    $lw_toman = irr_to_toman($k['last_week_sales_irr']);

    $summary = "در بازه زمانی {$features['range']['date_from']} تا {$features['range']['date_to']}، "
        . "فروش کل به " . fmt_toman($total_toman) . " رسید"
        . ($c['total_sales_pct'] !== null ? " و نسبت به دوره قبل " . fmt_pct($c['total_sales_pct']) . " تغییر کرد" : "")
        . ". میانگین فروش هر فروشگاه " . fmt_toman($avg_toman) . " است"
        . ($c['avg_sales_per_store_pct'] !== null ? " (" . fmt_pct($c['avg_sales_per_store_pct']) . ")" : "")
        . ". فروش هفته آخر " . fmt_toman($lw_toman)
        . ($c['last_week_sales_pct'] !== null ? " (" . fmt_pct($c['last_week_sales_pct']) . ")" : "")
        . ".";

    $cards = [];

    $cards[] = [
        "type"=>"insight",
        "severity"=>($c['total_sales_pct'] !== null && $c['total_sales_pct'] < 0 ? "warning" : "info"),
        "title"=>"روند فروش کل نسبت به دوره قبل",
        "text"=>"فروش کل این بازه " . fmt_toman($total_toman) . " است. "
            . ($c['total_sales_pct'] !== null ? "نسبت به دوره قبل " . fmt_pct($c['total_sales_pct']) . " تغییر کرده است. " : "")
            . "میانگین فروش هر فروشگاه " . fmt_toman($avg_toman)
            . ($c['avg_sales_per_store_pct'] !== null ? " (" . fmt_pct($c['avg_sales_per_store_pct']) . ")" : "")
            . " است.",
        "refs"=>["kpi","compare.kpi_prev"]
    ];

    $cards[] = [
        "type"=>"insight",
        "severity"=>($c['last_week_sales_pct'] !== null && $c['last_week_sales_pct'] > 0 ? "success" : "info"),
        "title"=>"سیگنال هفته آخر",
        "text"=>"فروش هفته آخر " . fmt_toman($lw_toman)
            . ($c['last_week_sales_pct'] !== null ? " و نسبت به دوره قبل " . fmt_pct($c['last_week_sales_pct']) . " تغییر کرده است." : ".")
            . " اگر فروش کل افت داشته اما هفته آخر رشد کرده باشد، احتمال «بازگشت روند» یا اثر تامین/پروموشن در انتهای دوره وجود دارد.",
        "refs"=>["kpi","compare.kpi_prev"]
    ];

    if ($c['stores_delta'] !== null && $c['stores_delta'] != 0) {
        $cards[] = [
            "type"=>"insight",
            "severity"=>($c['stores_delta'] < 0 ? "warning" : "info"),
            "title"=>"پوشش فروشگاه‌های فعال",
            "text"=>"تعداد فروشگاه‌های فعال از {$ctx['compare']['kpi_prev']['stores_count']} به {$k['stores_count']} رسیده است "
                . "({$c['stores_delta']}، " . ($c['stores_delta_pct'] !== null ? fmt_pct($c['stores_delta_pct']) : "نامشخص") . "). "
                . "بخشی از تغییر فروش می‌تواند ناشی از تغییر پوشش شبکه باشد نه فقط تقاضا.",
            "refs"=>["kpi","compare.kpi_prev"]
        ];
    }

    $cards[] = [
        "type"=>"insight",
        "severity"=>"info",
        "title"=>"تمرکز بازار در برندهای برتر",
        "text"=>"سهم 3 برند اول مجموعاً " . number_format($features['concentration']['top3_share_percent'], 2) . "% و "
            . "سهم 6 برند اول " . number_format($features['concentration']['top6_share_percent'], 2) . "% است. "
            . "تمرکز بالا یعنی چند برند/تامین‌کننده اثر قابل‌توجهی روی کل فروش دارند.",
        "refs"=>["top_brands_month"]
    ];

    if ($lb) {
        $cards[] = [
            "type"=>"insight",
            "severity"=>($lb['share_pp_change'] !== null && $lb['share_pp_change'] < 0 ? "warning" : "info"),
            "title"=>"وضعیت برند برتر",
            "text"=>$lb['brand_title'] . " با فروش " . fmt_toman(irr_to_toman($lb['sales_irr'])) . " سهم "
                . number_format($lb['share_percent'], 2) . "% دارد."
                . ($lb['share_pp_change'] !== null ? " تغییر سهم نسبت به دوره قبل: " . number_format($lb['share_pp_change'], 2) . " واحد درصد." : "")
                . " افت سهم می‌تواند نشانه از دست دادن قفسه/پروموشن باشد.",
            "refs"=>["top_brands_month","top_brands_prev"]
        ];
    }

    if ($ll1) {
        $cards[] = [
            "type"=>"insight",
            "severity"=>($ll1['share_pp_change'] !== null && $ll1['share_pp_change'] > 0 ? "info" : "warning"),
            "title"=>"گروه L1 با بیشترین سهم",
            "text"=>$ll1['l1_title'] . " با فروش " . fmt_toman(irr_to_toman($ll1['sales_irr'])) . " سهم "
                . number_format($ll1['share_percent'], 2) . "% دارد."
                . ($ll1['share_pp_change'] !== null ? " تغییر سهم: " . number_format($ll1['share_pp_change'], 2) . " واحد درصد." : "")
                . " اگر سهم رشد کند ولی فروش کل افت کند، این گروه «مقاوم‌تر از بازار» بوده است.",
            "refs"=>["l1_share","l1_share_prev"]
        ];
    }

    if ($ls) {
        $cards[] = [
            "type"=>"insight",
            "severity"=>($ls['sales_pct_change'] !== null && $ls['sales_pct_change'] < 0 ? "warning" : "info"),
            "title"=>"فروشگاه برتر و تغییر آن",
            "text"=>$ls['store_name'] . " با فروش " . fmt_toman(irr_to_toman($ls['sales_irr'])) . " در صدر است."
                . ($ls['sales_pct_change'] !== null ? " تغییر نسبت به دوره قبل: " . fmt_pct($ls['sales_pct_change']) . "." : "")
                . " اگر فروشگاه‌های برتر با هم افت کرده‌اند احتمال عامل سیستماتیک بیشتر است.",
            "refs"=>["top_stores","top_stores_prev"]
        ];
    }

    // ✅ New: last14 stability + spikes/outliers
    $spTxt = "";
    if (!empty($st['spikes'])) {
        $s = $st['spikes'][0];
        $spTxt = "بزرگ‌ترین جهش روزانه در ۱۴ روز آخر: {$s['date']} با تغییر " . fmt_pct($s['pct'])
            . " (از " . fmt_toman(irr_to_toman($s['prev'])) . " به " . fmt_toman(irr_to_toman($s['total_sales'])) . "). ";
    }
    $outTxt = "";
    if (!empty($st['outliers'])) {
        $o = $st['outliers'][0];
        $outTxt = "روز پرت محتمل: {$o['date']} با فروش " . fmt_toman(irr_to_toman($o['total_sales']))
            . " (Robust-Z=" . number_format($o['z'], 2) . "). ";
    }

    $cards[] = [
        "type"=>"insight",
        "severity"=>(($st['daily_total_sales_cv'] !== null && $st['daily_total_sales_cv'] > 1.0) ? "warning" : "info"),
        "title"=>"۱۴ روز آخر: نوسان، جهش‌ها و روزهای پرت",
        "text"=> $spTxt . $outTxt
            . "نوسان نسبی فروش روزانه (CV) حدود " . ($st['daily_total_sales_cv'] !== null ? number_format($st['daily_total_sales_cv']*100, 2) . "%" : "نامشخص")
            . " است. "
            . ($st['last7_vs_first7_pct'] !== null ? "میانگین ۷ روز آخر نسبت به ۷ روز اول " . fmt_pct($st['last7_vs_first7_pct']) . " تغییر کرده است. " : "")
            . "این کارت کمک می‌کند مدیر بفهمد «دقیقاً کِی» اتفاق غیرعادی افتاده.",
        "refs"=>["brand_share_14d"]
    ];

    $cards[] = [
        "type"=>"recommendation",
        "severity"=>"success",
        "title"=>"اقدام پیشنهادی (سریع و قابل اجرا)",
        "text"=>"1) اگر جهش/روز پرت داریم: علت همان تاریخ‌ها را پیدا کنید (تامین، پروموشن، قیمت، ثبت فاکتور، قطعی سیستم).\n"
            . "2) افت فروش را به دو بخش تفکیک کنید: «افت پوشش فروشگاه» vs «افت عملکرد» (فروش به ازای فروشگاه).\n"
            . "3) تغییر سهم برند/گروه‌های برتر را مبنای اولویت بگذارید: افت سهم = ریسک از دست دادن قفسه/موجودی.\n"
            . "4) 5 فروشگاه برتر را جداگانه مانیتور کنید؛ هم‌جهتی آنها معمولاً نشانه عامل سیستماتیک است.",
        "refs"=>["kpi","compare.kpi_prev","top_brands_month","l1_share","top_stores","brand_share_14d"]
    ];

    return [
        "summary" => $summary,
        "cards" => $cards,
        "kpis_used" => ["total_sales","avg_sales_per_store","last_week_sales","stores_count"],
        "charts_used" => ["top_brands_month","l1_share","top_stores","brand_share_14d"],
        "features_used" => $features,
    ];
}

// -----------------------------
// LLM prompt
// -----------------------------
$fallback_data = build_fallback($features_used, $ctx);

if (!$boot_ok) {
    // no config/helper => fallback
    respond($fallback_data, [
        "engine"=>"fallback",
        "model"=>null,
        "debug"=>[
            "llm_ok"=>false,
            "llm_error"=>$boot_err,
            "context_url"=>$context_url
        ],
        "filters"=>$ctx['filters'] ?? null,
        "generated_at"=>date('Y-m-d H:i:s'),
    ]);
}

$ai = ai_config(); // from src/config/ai.php
$base_url = $ai['https://gapgpt.app/api/v1/chat/completions'] ?? null;
$api_key  = $ai['sk-0C9w0UMNOWkXJY4uKASHrvuFGo2lzytwYrfd1EVOMGQGq77E'] ?? null;
$model    = $ai['model'] ?? 'gpt-4o-mini';

if (!$base_url || !$api_key) {
    respond($fallback_data, [
        "engine"=>"fallback",
        "model"=>$model,
        "debug"=>[
            "llm_ok"=>false,
            "llm_error"=>"GAPGPT_BASE_URL یا GAPGPT_API_KEY خالی است (env/runtime).",
            "base_url_present" => (bool)$base_url,
            "api_key_present"  => (bool)$api_key,
            "context_url"=>$context_url
        ],
        "filters"=>$ctx['filters'] ?? null,
        "generated_at"=>date('Y-m-d H:i:s'),
    ]);
}

$system = "You are a senior retail analytics assistant. Output MUST be valid JSON object only. No markdown. No extra text.";
$user = [
    "task" => "Generate executive-ready insights + actionable recommendations from provided structured retail context.",
    "requirements" => [
        "Output JSON with keys: summary, cards, kpis_used, charts_used.",
        "Cards should be concrete, decision-oriented, and reference available refs.",
        "Use last 14 days block specifically for short-term commentary; include spike/outlier interpretation.",
        "Avoid generic advice; be specific to numbers."
    ],
    "context" => [
        "filters" => $ctx['filters'] ?? null,
        "kpi" => $ctx['kpi'] ?? null,
        "overview" => [
            "top_brands_month" => $overview['top_brands_month'] ?? null,
            "l1_share" => $overview['l1_share'] ?? null,
            "top_stores" => $overview['top_stores'] ?? null,
            "brand_share_series_raw" => $overview['brand_share_14d'] ?? null,
            "short_term_last14_extracted" => $features_used['short_term_last14'] ?? null,
        ],
        "compare" => $ctx['compare'] ?? null,
    ],
    "output_schema" => [
        "summary" => "string",
        "cards" => [
            [
                "type"=>"insight|recommendation",
                "severity"=>"info|success|warning|danger",
                "title"=>"string",
                "text"=>"string",
                "refs"=>["string"]
            ]
        ],
        "kpis_used"=>["string"],
        "charts_used"=>["string"]
    ]
];

$payload = [
    "model" => $model,
    "temperature" => 0.2,
    "messages" => [
        ["role"=>"system","content"=>$system],
        ["role"=>"user","content"=>json_encode($user, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)]
    ]
];

// call GapGPT
try {
    $raw = gapgpt_chat($payload, $ai);
    $data = extract_json_from_text($raw);

    if (!is_array($data) || empty($data['summary']) || empty($data['cards'])) {
        // fallback if model didn’t obey
        respond($fallback_data, [
            "engine"=>"gapgpt",
            "model"=>$model,
            "debug"=>[
                "llm_ok"=>false,
                "llm_error"=>"LLM returned non-JSON or missing required keys.",
                "context_url"=>$context_url,
                "raw_preview"=>mb_substr((string)$raw, 0, 600)
            ],
            "filters"=>$ctx['filters'] ?? null,
            "generated_at"=>date('Y-m-d H:i:s'),
        ]);
    }

    // enrich with features_used (optional for debugging/dev)
    $data['features_used'] = $features_used;

    respond($data, [
        "engine"=>"gapgpt",
        "model"=>$model,
        "debug"=>[
            "llm_ok"=>true,
            "context_url"=>$context_url
        ],
        "filters"=>$ctx['filters'] ?? null,
        "generated_at"=>date('Y-m-d H:i:s'),
    ]);

} catch (Throwable $e) {
    respond($fallback_data, [
        "engine"=>"gapgpt",
        "model"=>$model,
        "debug"=>[
            "llm_ok"=>false,
            "llm_error"=>$e->getMessage(),
            "context_url"=>$context_url
        ],
        "filters"=>$ctx['filters'] ?? null,
        "generated_at"=>date('Y-m-d H:i:s'),
    ]);
}
