<?php
// /api/ai/context/dashboard.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

function _http_code_from_headers($headers) {
    if (!is_array($headers) || empty($headers[0])) return null;
    if (preg_match('#HTTP/\d\.\d\s+(\d{3})#', $headers[0], $m)) return (int)$m[1];
    return null;
}

function call_internal_api($path, $params = []) {
    $qs = http_build_query(array_filter($params, fn($v) => $v !== null && $v !== ''));
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';

    // بهتر از REQUEST_URI: SCRIPT_NAME همیشه مسیر فایل رو دقیق میده
    $script = $_SERVER['SCRIPT_NAME'] ?? '';
    // /dashboard-project/api/ai/context/dashboard.php  =>  /dashboard-project
    $basePath = preg_replace('#/api/ai/context/dashboard\.php$#', '', $script);
    $basePath = rtrim($basePath, '/');

    $url = "{$scheme}://{$host}{$basePath}/api/{$path}" . ($qs ? "?$qs" : "");

    $cookie = $_SERVER['HTTP_COOKIE'] ?? '';
    $auth   = $_SERVER['HTTP_AUTHORIZATION'] ?? ($_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? '');

    $headers = "Accept: application/json\r\n";
    if ($cookie) $headers .= "Cookie: {$cookie}\r\n";
    if ($auth)   $headers .= "Authorization: {$auth}\r\n";

    $ctx = stream_context_create([
        'http' => [
            'method'  => 'GET',
            'header'  => $headers,
            'timeout' => 25,
        ]
    ]);

    $raw = @file_get_contents($url, false, $ctx);
    $httpCode = _http_code_from_headers($http_response_header ?? null);

    if ($raw === false) {
        return ['ok' => false, 'error' => "Failed calling {$path}", 'url' => $url, 'http_code' => $httpCode, 'raw' => null];
    }

    $arr = json_decode($raw, true);
    if (!is_array($arr)) {
        return ['ok' => false, 'error' => "Invalid JSON from endpoint", 'url' => $url, 'http_code' => $httpCode, 'raw' => $raw];
    }

    // اگر endpoint خودش meta.ok داشت نگه داریم
    $metaOk = $arr['meta']['ok'] ?? true;

    return [
        'ok' => (bool)$metaOk,
        'url' => $url,
        'http_code' => $httpCode,
        'data' => $arr['data'] ?? null,
        'meta' => $arr['meta'] ?? null,
        'raw'  => null
    ];
}

function is_full_month_range($date_from, $date_to) {
    try {
        $df = new DateTime($date_from);
        $dt = new DateTime($date_to);

        // باید تو همون ماه باشه
        if ($df->format('Y-m') !== $dt->format('Y-m')) return false;

        $first = (clone $df)->modify('first day of this month')->format('Y-m-d');
        $last  = (clone $df)->modify('last day of this month')->format('Y-m-d');
        return ($df->format('Y-m-d') === $first) && ($dt->format('Y-m-d') === $last);
    } catch (Throwable $e) {
        return false;
    }
}

function prev_month_range($date_from) {
    $df = new DateTime($date_from);
    $prevFirst = (clone $df)->modify('first day of previous month')->format('Y-m-d');
    $prevLast  = (clone $df)->modify('last day of previous month')->format('Y-m-d');
    return [$prevFirst, $prevLast];
}

function prev_period_range($date_from, $date_to) {
    $df = new DateTime($date_from);
    $dt = new DateTime($date_to);

    $days = (int)$df->diff($dt)->format('%a'); // فاصله روزی
    // چون date_to inclusive گرفته میشه، +1
    $len = $days + 1;

    $prevTo = (clone $df)->modify('-1 day');
    $prevFrom = (clone $prevTo)->modify('-' . ($len - 1) . ' day');

    return [$prevFrom->format('Y-m-d'), $prevTo->format('Y-m-d')];
}

function yoy_range($date_from, $date_to) {
    $df = new DateTime($date_from);
    $dt = new DateTime($date_to);
    $df->modify('-1 year');
    $dt->modify('-1 year');
    return [$df->format('Y-m-d'), $dt->format('Y-m-d')];
}

function month_range_by_offset_from_start($date_from, int $offsetMonths) {
    // ماه مبنا: ماهِ date_from
    $df = new DateTime($date_from);
    $df->modify('first day of this month');
    $df->modify(($offsetMonths >= 0 ? '+' : '') . $offsetMonths . ' month');

    $from = (clone $df)->modify('first day of this month')->format('Y-m-d');
    $to   = (clone $df)->modify('last day of this month')->format('Y-m-d');
    return [$from, $to];
}

function avg3_months_kpi($baseParams, $date_from) {
    // میانگین ۳ ماه قبل از ماهِ date_from
    $sumSales = 0.0; $cntSales = 0;
    $sumAvg   = 0.0; $cntAvg = 0;
    $sumLW    = 0.0; $cntLW = 0;

    $periods = [];
    for ($i = 1; $i <= 3; $i++) {
        [$f, $t] = month_range_by_offset_from_start($date_from, -$i);
        $periods[] = ['date_from' => $f, 'date_to' => $t];

        $p = $baseParams;
        $p['date_from'] = $f;
        $p['date_to']   = $t;

        $k = call_internal_api('kpi/summary.php', $p);
        $d = $k['data'] ?? [];

        $sales = (float)($d['total_sales'] ?? 0);
        $avgps = (float)($d['avg_sales_per_store'] ?? 0);
        $lw    = (float)($d['last_week_sales'] ?? 0);

        // فقط اعداد معتبر رو لحاظ کن
        if ($sales > 0) { $sumSales += $sales; $cntSales++; }
        if ($avgps > 0) { $sumAvg += $avgps; $cntAvg++; }
        if ($lw > 0)    { $sumLW += $lw; $cntLW++; }
    }

    return [
        'periods' => $periods,
        'kpi_avg' => [
            'total_sales' => $cntSales ? (string)round($sumSales / $cntSales) : null,
            'avg_sales_per_store' => $cntAvg ? (string)round($sumAvg / $cntAvg) : null,
            'last_week_sales' => $cntLW ? (string)round($sumLW / $cntLW) : null,
        ]
    ];
}

function pct_delta($curr, $prev) {
    $c = (float)$curr;
    $p = (float)$prev;
    if ($p == 0) return null;
    return round((($c - $p) / $p) * 100, 2);
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $date_from = get_param('date_from');
    $date_to   = get_param('date_to');
    if (!$date_from || !$date_to) fail('date_from and date_to are required', 400);

    $brand_id = get_param('brand_id');
    $store_id = get_param('store_id');
    $city     = get_param('city');
    $area     = get_param('area');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');
    $province = get_param('province');

    // scope override
    if (!empty($scope['brand_id']))  $brand_id = (int)$scope['brand_id'];
    if (!empty($scope['province']))  $province = (string)$scope['province'];

    $compare = strtolower((string)get_param('compare', 'auto')); // auto|prev_month|prev_period|none
    if (!in_array($compare, ['auto','prev_month','prev_period','none'], true)) $compare = 'auto';

    // base filters
    $baseParams = [
        'date_from' => $date_from,
        'date_to'   => $date_to,
        'brand_id'  => $brand_id,
        'store_id'  => $store_id,
        'city'      => $city,
        'area'      => $area,
        'l1_id'     => $l1_id,
        'sale_type_id' => $sale_type_id,
        'province'  => $province,
    ];

    // ---------- Current period calls ----------
    $kpi       = call_internal_api('kpi/summary.php', $baseParams);
    $topBrands = call_internal_api('brands/top3m.php', array_merge($baseParams, ['per_page' => 6, 'page' => 1]));
    $l1Share   = call_internal_api('l1/top5.php', array_merge($baseParams, ['limit' => 5]));
    $brand14d  = call_internal_api('brands/daily_share.php', $baseParams);
    $topStores = call_internal_api('stores/top20.php', array_merge($baseParams, ['limit' => 20]));

    $data = [
        'user' => [
            'role' => $user['Role'] ?? null,
            'user_id' => $user['UserID'] ?? null,
            'username' => $user['Username'] ?? null,
        ],
        'scope' => [
            'province' => $scope['province'] ?? null,
            'brand_id' => $scope['brand_id'] ?? null,
        ],
        'filters' => [
            'date_from' => $date_from,
            'date_to'   => $date_to,
            'brand_id'  => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
            'store_id'  => ($store_id !== null && $store_id !== '') ? (int)$store_id : null,
            'city'      => $city ?: null,
            'area'      => $area ?: null,
            'l1_id'     => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
            'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
            'province'  => $province ?: null,
        ],
        'kpi' => $kpi['data'] ?? null,
        'overview' => [
            'top_brands_month' => $topBrands['data'] ?? null,
            'l1_share'         => $l1Share['data'] ?? null,
            'brand_share_14d'  => $brand14d['data'] ?? null,
            'top_stores'       => $topStores['data'] ?? null,
        ],
    ];

    // ---------- Compare ----------
    if ($compare !== 'none') {
        $mode = $compare;
        if ($mode === 'auto') {
            $mode = is_full_month_range($date_from, $date_to) ? 'prev_month' : 'prev_period';
        }

        // prev range
        if ($mode === 'prev_month') {
            [$p_from, $p_to] = prev_month_range($date_from);
        } else {
            [$p_from, $p_to] = prev_period_range($date_from, $date_to);
        }

        $prevParams = $baseParams;
        $prevParams['date_from'] = $p_from;
        $prevParams['date_to']   = $p_to;

        $kpi_prev       = call_internal_api('kpi/summary.php', $prevParams);
        $topBrands_prev = call_internal_api('brands/top3m.php', array_merge($prevParams, ['per_page' => 6, 'page' => 1]));
        $l1Share_prev   = call_internal_api('l1/top5.php', array_merge($prevParams, ['limit' => 5]));
        $brand14d_prev  = call_internal_api('brands/daily_share.php', $prevParams);
        $topStores_prev = call_internal_api('stores/top20.php', array_merge($prevParams, ['limit' => 20]));

        // YoY range
        [$y_from, $y_to] = yoy_range($date_from, $date_to);
        $yoyParams = $baseParams;
        $yoyParams['date_from'] = $y_from;
        $yoyParams['date_to']   = $y_to;

        $kpi_yoy = call_internal_api('kpi/summary.php', $yoyParams);

        $currK = $kpi['data'] ?? [];
        $prevK = $kpi_prev['data'] ?? [];
        $yoyK  = $kpi_yoy['data'] ?? [];

        $data['compare'] = [
            'mode' => $mode,

            'prev_range' => ['date_from' => $p_from, 'date_to' => $p_to],
            'kpi_prev' => $kpi_prev['data'] ?? null,
            'kpi_delta' => [
                'total_sales_pct' => pct_delta($currK['total_sales'] ?? 0, $prevK['total_sales'] ?? 0),
                'avg_sales_per_store_pct' => pct_delta($currK['avg_sales_per_store'] ?? 0, $prevK['avg_sales_per_store'] ?? 0),
                'last_week_sales_pct' => pct_delta($currK['last_week_sales'] ?? 0, $prevK['last_week_sales'] ?? 0),
            ],

            // YoY
            'same_period_last_year' => ['date_from' => $y_from, 'date_to' => $y_to],
            'kpi_yoy' => $kpi_yoy['data'] ?? null,
            'kpi_delta_yoy' => [
                'total_sales_pct' => pct_delta($currK['total_sales'] ?? 0, $yoyK['total_sales'] ?? 0),
                'avg_sales_per_store_pct' => pct_delta($currK['avg_sales_per_store'] ?? 0, $yoyK['avg_sales_per_store'] ?? 0),
                'last_week_sales_pct' => pct_delta($currK['last_week_sales'] ?? 0, $yoyK['last_week_sales'] ?? 0),
            ],

            // prev overview (optional, for deeper AI comparisons)
            'top_brands_prev' => $topBrands_prev['data'] ?? null,
            'l1_share_prev'   => $l1Share_prev['data'] ?? null,
            'brand_share_14d_prev' => $brand14d_prev['data'] ?? null,
            'top_stores_prev' => $topStores_prev['data'] ?? null,
        ];

        // last 3 months avg (only for full month range)
        if (is_full_month_range($date_from, $date_to)) {
            $data['compare']['last_3_months_avg'] = avg3_months_kpi($baseParams, $date_from);
        } else {
            $data['compare']['last_3_months_avg'] = null;
        }
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    // sources meta برای دیباگ (همون سبک خودت)
    $data['sources_meta'] = [
        'kpi' => $kpi,
        'topBrands' => $topBrands,
        'l1Share' => $l1Share,
        'brand14d' => $brand14d,
        'topStores' => $topStores,
    ];
    if (!empty($data['compare'])) {
        $data['sources_meta'] += [
            'kpi_prev' => $kpi_prev ?? null,
            'topBrands_prev' => $topBrands_prev ?? null,
            'l1Share_prev' => $l1Share_prev ?? null,
            'brand14d_prev' => $brand14d_prev ?? null,
            'topStores_prev' => $topStores_prev ?? null,
            'kpi_yoy' => $kpi_yoy ?? null,
        ];
    }

    // ✅ نکته کلیدی: json_response خودش data/meta را می‌سازد
    json_response([
  'data' => $data,
  'meta' => [
    'ok' => true,
    'generated_at' => date('Y-m-d H:i:s'),
    'duration_ms' => $elapsed_ms,
  ]
], 200);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
