<?php
// /api/admin/products/request_edit.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

function read_body_params(): array {
    // supports JSON and form-data
    $ct = strtolower((string)($_SERVER['CONTENT_TYPE'] ?? ''));
    if (strpos($ct, 'application/json') !== false) {
        $raw = file_get_contents('php://input');
        $j = json_decode($raw, true);
        return is_array($j) ? $j : [];
    }
    return $_POST ?: [];
}

function to_int_or_null($v): ?int {
    if ($v === null) return null;
    if ($v === '') return null;
    if (is_int($v)) return $v;
    if (is_string($v) && preg_match('/^\d+$/', $v)) return (int)$v;
    if (is_numeric($v)) return (int)$v;
    return null;
}

try {
    $t0 = microtime(true);

    $user = auth_user(); // must be logged-in

    $body = read_body_params();

    $product_id = to_int_or_null($body['product_id'] ?? null);
    if (!$product_id) {
        fail('product_id is required', 400);
    }

    $new_brand_id = to_int_or_null($body['new_brand_id'] ?? null);
    $new_l1_id    = to_int_or_null($body['new_l1_id'] ?? null);
    $new_l2_id    = to_int_or_null($body['new_l2_id'] ?? null);
    $new_l3_id    = to_int_or_null($body['new_l3_id'] ?? null);

    $note = trim((string)($body['note'] ?? ''));

    // حداقل یکی از فیلدهای جدید باید ارسال شده باشه
    if ($new_brand_id === null && $new_l1_id === null && $new_l2_id === null && $new_l3_id === null) {
        fail('At least one of new_brand_id/new_l1_id/new_l2_id/new_l3_id is required', 400);
    }

    // RequestedBy از کاربر لاگین‌شده
    $requested_by =
        (string)($user['Username'] ?? $user['UserName'] ?? $user['username'] ?? $user['email'] ?? $user['ID'] ?? 'user');

    $conn = db_connect();

    // SP: BI.sp_request_product_edit
    $sql = "EXEC BI.sp_request_product_edit
                @ProductID = ?,
                @NewBrandID = ?,
                @NewL1ID = ?,
                @NewL2ID = ?,
                @NewL3ID = ?,
                @RequestedBy = ?,
                @Note = ?";

    $params = [
        $product_id,
        $new_brand_id,
        $new_l1_id,
        $new_l2_id,
        $new_l3_id,
        $requested_by,
        ($note !== '' ? $note : null),
    ];

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        fail('DB error (request edit)', 500, [
            'sqlsrv_errors' => sqlsrv_errors(),
        ]);
    }

    $row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC) ?: [];
    $edit_id = $row['EditID'] ?? null;
    if ($edit_id === null) {
        // بعضی وقتا خروجی SP به هر دلیل برنمی‌گرده
        // اینجا با SELECT ScopeIdentity تو SP حل میشه که شما گذاشتی
        fail('EditID not returned from SP', 500);
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'edit_id' => (int)$edit_id,
        ],
        'meta' => [
            'ok' => true,
            'duration_ms' => $elapsed_ms,
            'source' => 'BI.sp_request_product_edit',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
