<?php
// /api/admin/products/pending.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

try {
    $t0 = microtime(true);

    $user = auth_user();
    $conn = db_connect();

    $status = trim((string)get_param('status', 'pending'));
    if ($status === '') $status = 'pending';

    $limit = (int)get_param('limit', 50);
    if ($limit <= 0) $limit = 50;
    if ($limit > 200) $limit = 200;

    $q = trim((string)get_param('q', '')); // optional search: product_id or text in note/requestedby

    $params = [];
    $where = "WHERE Status = ?";
    $params[] = $status;

    if ($q !== '') {
        if (preg_match('/^\d+$/', $q)) {
            $where .= " AND ProductID = ?";
            $params[] = (int)$q;
        } else {
            $where .= " AND (Note LIKE ? OR RequestedBy LIKE ?)";
            $like = '%' . $q . '%';
            $params[] = $like;
            $params[] = $like;
        }
    }

    $sql = "
        SELECT TOP ($limit)
            EditID,
            ProductID,
            OldBrandID, OldL1ID, OldL2ID, OldL3ID,
            NewBrandID, NewL1ID, NewL2ID, NewL3ID,
            Status,
            Note,
            RequestedBy,
            RequestedAt,
            ReviewedBy,
            ReviewedAt,
            AutoApplyAt,
            AppliedAt,
            ApplyError
        FROM BI.ProductEditsQueue
        $where
        ORDER BY RequestedAt DESC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        fail('DB error (pending list)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    }

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'edit_id' => (int)$r['EditID'],
            'product_id' => (int)$r['ProductID'],
            'old' => [
                'brand_id' => $r['OldBrandID'] !== null ? (int)$r['OldBrandID'] : null,
                'l1_id' => $r['OldL1ID'] !== null ? (int)$r['OldL1ID'] : null,
                'l2_id' => $r['OldL2ID'] !== null ? (int)$r['OldL2ID'] : null,
                'l3_id' => $r['OldL3ID'] !== null ? (int)$r['OldL3ID'] : null,
            ],
            'new' => [
                'brand_id' => $r['NewBrandID'] !== null ? (int)$r['NewBrandID'] : null,
                'l1_id' => $r['NewL1ID'] !== null ? (int)$r['NewL1ID'] : null,
                'l2_id' => $r['NewL2ID'] !== null ? (int)$r['NewL2ID'] : null,
                'l3_id' => $r['NewL3ID'] !== null ? (int)$r['NewL3ID'] : null,
            ],
            'status' => (string)$r['Status'],
            'note' => (string)($r['Note'] ?? ''),
            'requested_by' => (string)($r['RequestedBy'] ?? ''),
            'requested_at' => $r['RequestedAt'] ? $r['RequestedAt']->format('Y-m-d H:i:s') : null,
            'reviewed_by' => (string)($r['ReviewedBy'] ?? ''),
            'reviewed_at' => $r['ReviewedAt'] ? $r['ReviewedAt']->format('Y-m-d H:i:s') : null,
            'auto_apply_at' => $r['AutoApplyAt'] ? $r['AutoApplyAt']->format('Y-m-d H:i:s') : null,
            'applied_at' => $r['AppliedAt'] ? $r['AppliedAt']->format('Y-m-d H:i:s') : null,
            'apply_error' => (string)($r['ApplyError'] ?? ''),
        ];
    }

    json_response([
        'data' => [
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'filters' => [
                'status' => $status,
                'q' => $q ?: null,
                'limit' => $limit
            ],
            'duration_ms' => (int)round((microtime(true) - $t0) * 1000),
            'source' => 'BI.ProductEditsQueue',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
