<?php
// /api/admin/products/get.php  (FAST + fallback to SaleDetail for missing ProductMaster rows)
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

try {
    $t0 = microtime(true);

    $user = auth_user();
    $conn = db_connect();

    $product_id = get_param('product_id');
    if ($product_id === null || $product_id === '' || !preg_match('/^\d+$/', (string)$product_id)) {
        fail('product_id is required (numeric)', 400);
    }
    $product_id = (int)$product_id;

    // 1) Try ProductMaster first (fast)
    $sqlPM = "
        SELECT TOP (1)
            pm.ProductID AS product_id,
            pm.ProductTitle AS product_title,

            pm.BrandID AS base_brand_id,
            pm.BrandTitle AS base_brand_title,
            pm.L1ID AS base_l1_id,
            pm.L1Title AS base_l1_title,
            pm.L2ID AS base_l2_id,
            pm.L2Title AS base_l2_title,
            pm.L3ID AS base_l3_id,
            pm.L3Title AS base_l3_title,

            oa.ActiveVersion AS override_version,

            oh.BrandID AS ov_brand_id,
            oh.BrandTitle AS ov_brand_title,
            oh.L1ID AS ov_l1_id,
            oh.L1Title AS ov_l1_title,
            oh.L2ID AS ov_l2_id,
            oh.L2Title AS ov_l2_title,
            oh.L3ID AS ov_l3_id,
            oh.L3Title AS ov_l3_title,

            oh.SourceEditID AS source_edit_id,
            oh.Note AS override_note,
            oh.CreatedBy AS override_created_by,
            oh.CreatedAt AS override_created_at
        FROM BI.ProductMaster pm
        LEFT JOIN BI.ProductOverrideActive oa
            ON oa.ProductID = pm.ProductID
        LEFT JOIN BI.ProductOverrideHistory oh
            ON oh.ProductID = oa.ProductID AND oh.VersionNo = oa.ActiveVersion
        WHERE pm.ProductID = ?
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sqlPM, [$product_id]);
    if ($stmt === false) {
        fail('DB error (product get pm)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    }
    $r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);

    // 2) Fallback: read one row from SaleDetail (still cheap: by ProductID + TOP 1)
    if (!$r) {
        $sqlSD = "
            SELECT TOP (1)
                sd.ProductID AS product_id,
                sd.ProductTitle AS product_title,

                sd.BrandID AS base_brand_id,
                sd.BrandTitle AS base_brand_title,
                sd.ProductGroupID AS base_l1_id,
                sd.ProductGroupTitle AS base_l1_title,
                sd.ProductCategoryID AS base_l2_id,
                sd.ProductCategoryTitle AS base_l2_title,
                sd.ProductSubCategoryID AS base_l3_id,
                sd.ProductSubCategoryTitle AS base_l3_title,

                oa.ActiveVersion AS override_version,

                oh.BrandID AS ov_brand_id,
                oh.BrandTitle AS ov_brand_title,
                oh.L1ID AS ov_l1_id,
                oh.L1Title AS ov_l1_title,
                oh.L2ID AS ov_l2_id,
                oh.L2Title AS ov_l2_title,
                oh.L3ID AS ov_l3_id,
                oh.L3Title AS ov_l3_title,

                oh.SourceEditID AS source_edit_id,
                oh.Note AS override_note,
                oh.CreatedBy AS override_created_by,
                oh.CreatedAt AS override_created_at
            FROM BI.SaleDetail sd WITH (NOLOCK)
            LEFT JOIN BI.ProductOverrideActive oa
                ON oa.ProductID = sd.ProductID
            LEFT JOIN BI.ProductOverrideHistory oh
                ON oh.ProductID = oa.ProductID AND oh.VersionNo = oa.ActiveVersion
            WHERE sd.ProductID = ?
            ORDER BY sd.FactorDate DESC
            OPTION (RECOMPILE);
        ";

        $stmt2 = sqlsrv_query($conn, $sqlSD, [$product_id]);
        if ($stmt2 === false) {
            fail('DB error (product get sd)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        }
        $r = sqlsrv_fetch_array($stmt2, SQLSRV_FETCH_ASSOC);

        if (!$r) {
            fail('Product not found', 404, ['product_id' => $product_id]);
        }

        $source = 'BI.SaleDetail (fallback) + Override';
    } else {
        $source = 'BI.ProductMaster (+Override)';
    }

    $is_overridden = ($r['override_version'] !== null);

    $base = [
        'brand_id' => $r['base_brand_id'] !== null ? (int)$r['base_brand_id'] : null,
        'brand_title' => (string)($r['base_brand_title'] ?? ''),
        'l1_id' => $r['base_l1_id'] !== null ? (int)$r['base_l1_id'] : null,
        'l1_title' => (string)($r['base_l1_title'] ?? ''),
        'l2_id' => $r['base_l2_id'] !== null ? (int)$r['base_l2_id'] : null,
        'l2_title' => (string)($r['base_l2_title'] ?? ''),
        'l3_id' => $r['base_l3_id'] !== null ? (int)$r['base_l3_id'] : null,
        'l3_title' => (string)($r['base_l3_title'] ?? ''),
    ];

    $override = null;
    if ($is_overridden) {
        $override = [
            'version' => (int)$r['override_version'],
            'brand_id' => $r['ov_brand_id'] !== null ? (int)$r['ov_brand_id'] : null,
            'brand_title' => (string)($r['ov_brand_title'] ?? ''),
            'l1_id' => $r['ov_l1_id'] !== null ? (int)$r['ov_l1_id'] : null,
            'l1_title' => (string)($r['ov_l1_title'] ?? ''),
            'l2_id' => $r['ov_l2_id'] !== null ? (int)$r['ov_l2_id'] : null,
            'l2_title' => (string)($r['ov_l2_title'] ?? ''),
            'l3_id' => $r['ov_l3_id'] !== null ? (int)$r['ov_l3_id'] : null,
            'l3_title' => (string)($r['ov_l3_title'] ?? ''),
            'source_edit_id' => $r['source_edit_id'] !== null ? (int)$r['source_edit_id'] : null,
            'note' => (string)($r['override_note'] ?? ''),
            'created_by' => (string)($r['override_created_by'] ?? ''),
            'created_at' => $r['override_created_at'] ? $r['override_created_at']->format('Y-m-d H:i:s') : null,
        ];
    }

    $effective = [
        'brand_id' => $is_overridden ? $override['brand_id'] : $base['brand_id'],
        'brand_title' => $is_overridden ? $override['brand_title'] : $base['brand_title'],
        'l1_id' => $is_overridden ? $override['l1_id'] : $base['l1_id'],
        'l1_title' => $is_overridden ? $override['l1_title'] : $base['l1_title'],
        'l2_id' => $is_overridden ? $override['l2_id'] : $base['l2_id'],
        'l2_title' => $is_overridden ? $override['l2_title'] : $base['l2_title'],
        'l3_id' => $is_overridden ? $override['l3_id'] : $base['l3_id'],
        'l3_title' => $is_overridden ? $override['l3_title'] : $base['l3_title'],
    ];

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'product_id' => (int)$r['product_id'],
            'product_title' => (string)($r['product_title'] ?? ''),
            'is_overridden' => $is_overridden,
            'base' => $base,
            'override' => $override,
            'effective' => $effective,
        ],
        'meta' => [
            'ok' => true,
            'filters' => ['product_id' => $product_id],
            'duration_ms' => $elapsed_ms,
            'source' => $source,
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
