<?php
// /src/helpers/GapGPT.php

require_once __DIR__ . '/../config/ai.php';

/**
 * Minimal JSON POST client.
 */
function http_post_json(string $url, array $payload, array $headers = [], int $timeout_s = 45): array {
    $body = json_encode($payload, JSON_UNESCAPED_UNICODE);

    $h = array_merge([
        'Content-Type: application/json',
        'Accept: application/json',
    ], $headers);

    // Prefer cURL when available
    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $h);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout_s);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, min(10, $timeout_s));

        $raw = curl_exec($ch);
        $errno = curl_errno($ch);
        $err  = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return [
            'ok' => ($errno === 0) && ($code >= 200 && $code < 300) && $raw !== false,
            'http_code' => $code,
            'raw' => ($raw === false ? '' : (string)$raw),
            'error' => ($errno === 0 ? null : $err),
        ];
    }

    // Fallback: file_get_contents
    $ctx = stream_context_create([
        'http' => [
            'method'  => 'POST',
            'timeout' => $timeout_s,
            'header'  => implode("\r\n", $h),
            'content' => $body,
        ]
    ]);

    $raw = @file_get_contents($url, false, $ctx);
    $code = 0;
    if (isset($http_response_header) && is_array($http_response_header)) {
        foreach ($http_response_header as $line) {
            if (preg_match('~HTTP/\S+\s+(\d+)~', $line, $m)) { $code = (int)$m[1]; break; }
        }
    }

    return [
        'ok' => ($raw !== false) && ($code >= 200 && $code < 300),
        'http_code' => $code,
        'raw' => ($raw === false ? '' : (string)$raw),
        'error' => ($raw === false ? 'HTTP request failed' : null),
    ];
}

/**
 * Calls GapGPT chat endpoint.
 * Tries to be compatible with OpenAI-style chat-completions.
 */
function gapgpt_chat(array $messages, array $opts = []): array {
    $cfg = ai_config();
    $key = $cfg['api_key'];
    $url = $cfg['base_url'];

    if (!$key) {
        return ['ok' => false, 'error' => 'Missing GAPGPT_API_KEY', 'http_code' => 0, 'raw' => null];
    }

    $payload = [
        'model' => $opts['model'] ?? $cfg['model'],
        'messages' => $messages,
        'temperature' => $opts['temperature'] ?? 0.2,
        'max_tokens' => $opts['max_tokens'] ?? 900,
    ];

    // Optional knobs
    if (isset($opts['top_p'])) $payload['top_p'] = $opts['top_p'];

    $res = http_post_json($url, $payload, [
        'Authorization: Bearer ' . $key,
    ], (int)($opts['timeout_s'] ?? $cfg['timeout_s']));

    if (!$res['ok']) {
        return $res + ['error' => $res['error'] ?? 'GapGPT request failed'];
    }

    $arr = json_decode($res['raw'], true);
    if (!is_array($arr)) {
        return ['ok' => false, 'http_code' => $res['http_code'], 'raw' => $res['raw'], 'error' => 'Invalid JSON from GapGPT'];
    }

    // Try OpenAI-like shapes
    $content = null;
    if (isset($arr['choices'][0]['message']['content'])) {
        $content = $arr['choices'][0]['message']['content'];
    } elseif (isset($arr['data']['choices'][0]['message']['content'])) {
        $content = $arr['data']['choices'][0]['message']['content'];
    } elseif (isset($arr['content'])) {
        $content = $arr['content'];
    }

    return ['ok' => true, 'http_code' => $res['http_code'], 'raw' => $arr, 'content' => $content];
}

/**
 * Extract JSON object/array from a model output.
 */
function extract_json_from_text(string $text): ?array {
    $text = trim($text);
    if ($text === '') return null;

    // Direct parse
    $arr = json_decode($text, true);
    if (is_array($arr)) return $arr;

    // Try to find first {...} block
    $start = strpos($text, '{');
    $end   = strrpos($text, '}');
    if ($start !== false && $end !== false && $end > $start) {
        $candidate = substr($text, $start, $end - $start + 1);
        $arr = json_decode($candidate, true);
        if (is_array($arr)) return $arr;
    }

    // Try to find first [...] block
    $start = strpos($text, '[');
    $end   = strrpos($text, ']');
    if ($start !== false && $end !== false && $end > $start) {
        $candidate = substr($text, $start, $end - $start + 1);
        $arr = json_decode($candidate, true);
        if (is_array($arr)) return $arr;
    }

    return null;
}
