<?php
// /src/helpers/AIClient.php

require_once __DIR__ . '/../config/ai.php';

/**
 * Minimal JSON HTTP client using cURL.
 */
function http_json(string $method, string $url, array $headers = [], $body = null, int $timeout = 30): array {
    $ch = curl_init($url);
    if ($ch === false) return ['ok' => false, 'error' => 'Failed to init curl'];

    $hdrs = [];
    foreach ($headers as $k => $v) {
        $hdrs[] = $k . ': ' . $v;
    }

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST  => strtoupper($method),
        CURLOPT_HTTPHEADER     => $hdrs,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_CONNECTTIMEOUT => min(10, $timeout),
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    if ($body !== null) {
        $payload = is_string($body) ? $body : json_encode($body, JSON_UNESCAPED_UNICODE);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    }

    $raw = curl_exec($ch);
    $errno = curl_errno($ch);
    $err   = curl_error($ch);
    $code  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($raw === false) {
        return ['ok' => false, 'http_code' => $code, 'error' => 'curl_error: ' . ($err ?: ('errno ' . $errno))];
    }

    $decoded = json_decode($raw, true);
    if (!is_array($decoded)) {
        return ['ok' => false, 'http_code' => $code, 'error' => 'Invalid JSON response', 'raw' => $raw];
    }

    return ['ok' => ($code >= 200 && $code < 300), 'http_code' => $code, 'data' => $decoded, 'raw' => $raw];
}

/**
 * Call GapGPT Chat Completions (OpenAI-compatible shape).
 * Returns: ['ok'=>bool, 'content'=>string|null, 'data'=>array|null, 'http_code'=>int]
 */
function gapgpt_chat(array $messages, array $opts = []): array {
    $cfg = gapgpt_config();

    $apiKey = $opts['api_key'] ?? $cfg['api_key'];
    if (!$apiKey) {
        return ['ok' => false, 'error' => 'Missing GAPGPT_API_KEY (set env var or config)', 'http_code' => 0];
    }

    $url = $opts['base_url'] ?? $cfg['base_url'];
    $model = $opts['model'] ?? $cfg['model'];

    $payload = [
        'model' => $model,
        'messages' => $messages,
        'temperature' => $opts['temperature'] ?? $cfg['temperature'],
        'max_tokens' => $opts['max_tokens'] ?? $cfg['max_tokens'],
    ];

    // If your provider supports JSON mode, you can pass it via opts['response_format']
    if (isset($opts['response_format'])) {
        $payload['response_format'] = $opts['response_format'];
    }

    $res = http_json('POST', $url, [
        'Authorization' => 'Bearer ' . $apiKey,
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json',
    ], $payload, (int)($opts['timeout_sec'] ?? $cfg['timeout_sec']));

    if (!$res['ok']) {
        return ['ok' => false, 'error' => $res['error'] ?? 'Request failed', 'http_code' => $res['http_code'] ?? 0, 'data' => $res['data'] ?? null, 'raw' => $res['raw'] ?? null];
    }

    $data = $res['data'];

    // OpenAI-like: choices[0].message.content
    $content = null;
    if (isset($data['choices'][0]['message']['content'])) {
        $content = (string)$data['choices'][0]['message']['content'];
    } elseif (isset($data['data']['choices'][0]['message']['content'])) {
        // some wrappers nest under data
        $content = (string)$data['data']['choices'][0]['message']['content'];
    }

    return ['ok' => true, 'http_code' => $res['http_code'], 'content' => $content, 'data' => $data];
}

/**
 * Try to decode JSON from a model text.
 */
function ai_extract_json(string $text): ?array {
    $text = trim($text);
    if ($text === '') return null;

    // Strip markdown fences if present
    $text = preg_replace('/^```(json)?/i', '', $text);
    $text = preg_replace('/```$/', '', $text);
    $text = trim($text);

    $j = json_decode($text, true);
    if (is_array($j)) return $j;

    // Try to extract the first {...} block
    $start = strpos($text, '{');
    $end   = strrpos($text, '}');
    if ($start === false || $end === false || $end <= $start) return null;

    $candidate = substr($text, $start, $end - $start + 1);
    $j2 = json_decode($candidate, true);
    return is_array($j2) ? $j2 : null;
}
<?php
// /src/helpers/AIClient.php

require_once __DIR__ . '/../config/ai.php';

/**
 * Minimal JSON HTTP client using cURL.
 */
function http_json(string $method, string $url, array $headers = [], $body = null, int $timeout = 30): array {
    $ch = curl_init($url);
    if ($ch === false) return ['ok' => false, 'error' => 'Failed to init curl'];

    $hdrs = [];
    foreach ($headers as $k => $v) {
        $hdrs[] = $k . ': ' . $v;
    }

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST  => strtoupper($method),
        CURLOPT_HTTPHEADER     => $hdrs,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_CONNECTTIMEOUT => min(10, $timeout),
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    if ($body !== null) {
        $payload = is_string($body) ? $body : json_encode($body, JSON_UNESCAPED_UNICODE);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    }

    $raw = curl_exec($ch);
    $errno = curl_errno($ch);
    $err   = curl_error($ch);
    $code  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($raw === false) {
        return ['ok' => false, 'http_code' => $code, 'error' => 'curl_error: ' . ($err ?: ('errno ' . $errno))];
    }

    $decoded = json_decode($raw, true);
    if (!is_array($decoded)) {
        return ['ok' => false, 'http_code' => $code, 'error' => 'Invalid JSON response', 'raw' => $raw];
    }

    return ['ok' => ($code >= 200 && $code < 300), 'http_code' => $code, 'data' => $decoded, 'raw' => $raw];
}

/**
 * Call GapGPT Chat Completions (OpenAI-compatible shape).
 * Returns: ['ok'=>bool, 'content'=>string|null, 'data'=>array|null, 'http_code'=>int]
 */
function gapgpt_chat(array $messages, array $opts = []): array {
    $cfg = gapgpt_config();

    $apiKey = $opts['api_key'] ?? $cfg['api_key'];
    if (!$apiKey) {
        return ['ok' => false, 'error' => 'Missing GAPGPT_API_KEY (set env var or config)', 'http_code' => 0];
    }

    $url = $opts['base_url'] ?? $cfg['base_url'];
    $model = $opts['model'] ?? $cfg['model'];

    $payload = [
        'model' => $model,
        'messages' => $messages,
        'temperature' => $opts['temperature'] ?? $cfg['temperature'],
        'max_tokens' => $opts['max_tokens'] ?? $cfg['max_tokens'],
    ];

    // If your provider supports JSON mode, you can pass it via opts['response_format']
    if (isset($opts['response_format'])) {
        $payload['response_format'] = $opts['response_format'];
    }

    $res = http_json('POST', $url, [
        'Authorization' => 'Bearer ' . $apiKey,
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json',
    ], $payload, (int)($opts['timeout_sec'] ?? $cfg['timeout_sec']));

    if (!$res['ok']) {
        return ['ok' => false, 'error' => $res['error'] ?? 'Request failed', 'http_code' => $res['http_code'] ?? 0, 'data' => $res['data'] ?? null, 'raw' => $res['raw'] ?? null];
    }

    $data = $res['data'];

    // OpenAI-like: choices[0].message.content
    $content = null;
    if (isset($data['choices'][0]['message']['content'])) {
        $content = (string)$data['choices'][0]['message']['content'];
    } elseif (isset($data['data']['choices'][0]['message']['content'])) {
        // some wrappers nest under data
        $content = (string)$data['data']['choices'][0]['message']['content'];
    }

    return ['ok' => true, 'http_code' => $res['http_code'], 'content' => $content, 'data' => $data];
}

/**
 * Try to decode JSON from a model text.
 */
function ai_extract_json(string $text): ?array {
    $text = trim($text);
    if ($text === '') return null;

    // Strip markdown fences if present
    $text = preg_replace('/^```(json)?/i', '', $text);
    $text = preg_replace('/```$/', '', $text);
    $text = trim($text);

    $j = json_decode($text, true);
    if (is_array($j)) return $j;

    // Try to extract the first {...} block
    $start = strpos($text, '{');
    $end   = strrpos($text, '}');
    if ($start === false || $end === false || $end <= $start) return null;

    $candidate = substr($text, $start, $end - $start + 1);
    $j2 = json_decode($candidate, true);
    return is_array($j2) ? $j2 : null;
}
