<?php
// /api/stores/top3m.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';
require_once __DIR__ . '/../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    // dates
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // default: last 90 days
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    // optional filters
    $store_id = get_param('store_id');
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // role scope
    if (!empty($scope['province'])) $province = $scope['province'];
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $where = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($store_id !== null && $store_id !== '') { $where .= " AND a.StoreID = ?"; $params[] = (int)$store_id; }
    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND a.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // DimStore filters (string match با trim)
    if ($province !== '') { $where .= " AND LTRIM(RTRIM(ds.Province)) = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND LTRIM(RTRIM(ds.City)) = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND LTRIM(RTRIM(CAST(ds.Area AS nvarchar(50)))) = ?"; $params[] = $area; }

    $conn = db_connect();

    // اگر store_id داده شده باشد: همان را برگردان، وگرنه top1
    $top = ($store_id !== null && $store_id !== '') ? "" : "TOP 1";

    $sql = "
        SELECT $top
            a.StoreID AS store_id,
            MAX(ds.StoreName) AS store_name,
            MAX(ds.Province)  AS province,
            MAX(ds.City)      AS city,
            MAX(CAST(ds.Area AS nvarchar(50))) AS area,
            MAX(ds.YaranCode) AS yaran_code,
            MAX(ds.YaranName) AS yaran_name,
            MAX(CAST(ds.Lat_Long AS nvarchar(100))) AS lat_long,
            CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount))) AS sales_amount
        FROM BI.AggDailyStoreBrandL1 a
        JOIN BI.DimStore ds ON ds.StoreID = a.StoreID
        $where
        GROUP BY a.StoreID
        " . (($store_id !== null && $store_id !== '') ? "" : "ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC") . "
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) fail('DB error (stores top3m)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);
    if (!$row) {
        json_response([
            'data' => null,
            'meta' => [
                'ok' => true,
                'empty' => true,
                'filters' => [
                    'date_from' => $date_from,
                    'date_to'   => $date_to,
                    'province'  => $province ?: null,
                    'city'      => $city ?: null,
                    'area'      => $area ?: null,
                    'store_id'  => $store_id !== '' ? ($store_id !== null ? (int)$store_id : null) : null,
                    'brand_id'  => $brand_id !== '' ? ($brand_id !== null ? (int)$brand_id : null) : null,
                    'l1_id'     => $l1_id !== '' ? ($l1_id !== null ? (int)$l1_id : null) : null,
                    'sale_type_id' => $sale_type_id !== '' ? ($sale_type_id !== null ? (int)$sale_type_id : null) : null,
                ],
            ],
        ]);
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => $row,
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'province'  => $province ?: null,
                'city'      => $city ?: null,
                'area'      => $area ?: null,
                'store_id'  => $store_id !== '' ? ($store_id !== null ? (int)$store_id : null) : null,
                'brand_id'  => $brand_id !== '' ? ($brand_id !== null ? (int)$brand_id : null) : null,
                'l1_id'     => $l1_id !== '' ? ($l1_id !== null ? (int)$l1_id : null) : null,
                'sale_type_id' => $sale_type_id !== '' ? ($sale_type_id !== null ? (int)$sale_type_id : null) : null,
            ],
            'duration_ms' => $elapsed_ms,
            'selected_mode' => ($store_id !== null && $store_id !== '') ? 'manual_store_id' : 'top_store_3m',
            'amount_unit' => 'IRR',
            'source' => 'BI.AggDailyStoreBrandL1 + BI.DimStore',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
