<?php
// /api/stores/selected/top_products_l2.php  (Source: BI.SaleDetail)
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    $l2_id    = get_param('l2_id');

    if ($store_id === null || $store_id === '') fail('store_id is required', 400);
    if ($l2_id === null || $l2_id === '') fail('l2_id is required', 400);

    $store_id = (int)$store_id;
    $l2_id    = (int)$l2_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $sale_type_id = get_param('sale_type_id');

    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $limit = (int)get_param('limit', 5);
    if ($limit <= 0) $limit = 5;
    if ($limit > 50) $limit = 50;

    $conn = db_connect();

    // province manager scope check
    if (!empty($scope['province'])) {
        $stChk = sqlsrv_query($conn, "SELECT TOP 1 Province FROM BI.DimStore WHERE StoreID = ?", [$store_id]);
        if ($stChk === false) fail('DB error (store scope check)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        $rChk = sqlsrv_fetch_array($stChk, SQLSRV_FETCH_ASSOC);
        if (!$rChk || trim((string)$rChk['Province']) !== trim((string)$scope['province'])) {
            json_response(['data' => [], 'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province']]);
        }
    }

    $where = "WHERE sd.StoreID = ? AND sd.FactorDate >= ? AND sd.FactorDate < ? AND sd.ProductCategoryID = ?";
    $params = [$store_id, $date_from, $date_to_excl, $l2_id];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND sd.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    $sql = "
        SELECT TOP ($limit)
            sd.ProductID AS product_id,
            MAX(sd.ProductTitle) AS product_title,
            sd.BrandID AS brand_id,
            MAX(sd.BrandTitle) AS brand_title,
            CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice))) AS sales_amount
        FROM BI.SaleDetail sd WITH (NOLOCK)
        $where
          AND sd.ProductID IS NOT NULL
        GROUP BY sd.ProductID, sd.BrandID
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) fail('DB error (top products l2 from SaleDetail)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'product_id' => (int)$r['product_id'],
            'product_title' => (string)($r['product_title'] ?? ''),
            'brand_id' => (int)$r['brand_id'],
            'brand_title' => (string)($r['brand_title'] ?? ''),
            'sales_amount' => (string)($r['sales_amount'] ?? '0'),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'l2_id' => $l2_id,
            'items' => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => ['province' => $scope['province'], 'brand_id' => $scope['brand_id']],
            'filters' => [
                'store_id' => $store_id,
                'l2_id' => $l2_id,
                'date_from' => $date_from,
                'date_to' => $date_to,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
                'limit' => $limit,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.SaleDetail (Store-scoped)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
