<?php
// /api/stores/selected/top_brands.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') {
        fail('store_id is required', 400);
    }
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $l1_id = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    $limit = (int)get_param('limit', 5);
    if ($limit <= 0) $limit = 5;
    if ($limit > 20) $limit = 20;

    $brand_id = get_param('brand_id');
    // role scope
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $conn = db_connect();

    // province manager scope check
    if (!empty($scope['province'])) {
        $stChk = sqlsrv_query($conn, "SELECT TOP 1 Province FROM BI.DimStore WHERE StoreID = ?", [$store_id]);
        if ($stChk === false) fail('DB error (store scope check)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        $rChk = sqlsrv_fetch_array($stChk, SQLSRV_FETCH_ASSOC);
        if (!$rChk || trim((string)$rChk['Province']) !== trim((string)$scope['province'])) {
            json_response(['data' => [], 'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province']]);
        }
    }

    $where = "WHERE a.StoreID = ? AND a.DateKey >= ? AND a.DateKey < ?";
    $params = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND a.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // اگر برند منیجر باشد، limit بی‌معنی می‌شود (فقط همان برند)
    $top = ($brand_id !== null && $brand_id !== '') ? "" : "TOP ($limit)";

    $sql = "
        SELECT $top
            a.BrandID AS brand_id,
            MAX(db.BrandTitle) AS brand_title,
            CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount))) AS sales_amount
        FROM BI.AggDailyStoreBrandL1 a
        LEFT JOIN BI.DimBrand db ON db.BrandID = a.BrandID
        $where
        GROUP BY a.BrandID
        " . (($brand_id !== null && $brand_id !== '') ? "" : "ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC") . "
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        // fallback بدون DimBrand
        $sql2 = "
            SELECT $top
                a.BrandID AS brand_id,
                CAST(a.BrandID AS nvarchar(50)) AS brand_title,
                CONVERT(varchar(60), SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount))) AS sales_amount
            FROM BI.AggDailyStoreBrandL1 a
            $where
            GROUP BY a.BrandID
            " . (($brand_id !== null && $brand_id !== '') ? "" : "ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC") . "
        ";
        $stmt = sqlsrv_query($conn, $sql2, $params);
        if ($stmt === false) fail('DB error (top brands)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    }

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'brand_id' => isset($r['brand_id']) ? (int)$r['brand_id'] : null,
            'brand_title' => (string)($r['brand_title'] ?? ''),
            'sales_amount' => (string)($r['sales_amount'] ?? '0'),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'items' => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'store_id' => $store_id,
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id'  => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'     => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
                'limit' => $limit,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'AggDailyStoreBrandL1 (fallback DimBrand)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
