<?php
// /api/stores/selected/l1_share.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') {
        fail('store_id is required', 400);
    }
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $conn = db_connect();

    // ensure store in scope for province manager
    if (!empty($scope['province'])) {
        $stChk = sqlsrv_query($conn, "SELECT TOP 1 Province FROM BI.DimStore WHERE StoreID = ?", [$store_id]);
        if ($stChk === false) fail('DB error (store scope check)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
        $rChk = sqlsrv_fetch_array($stChk, SQLSRV_FETCH_ASSOC);
        if (!$rChk || trim((string)$rChk['Province']) !== trim((string)$scope['province'])) {
            json_response(['data' => [], 'meta' => ['ok' => true, 'empty' => true, 'reason' => 'store_out_of_scope_province']]);
        }
    }

    $where = "WHERE ap.StoreID = ? AND ap.DateKey >= ? AND ap.DateKey < ?";
    $params = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND ap.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND ap.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // total for share
    $sqlTotal = "SELECT SUM(TRY_CONVERT(decimal(38,0), ap.SalesAmount)) AS total_sales
                 FROM BI.AggDailyProduct ap
                 $where";
    $stT = sqlsrv_query($conn, $sqlTotal, $params);
    if ($stT === false) fail('DB error (l1 share total)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rt = sqlsrv_fetch_array($stT, SQLSRV_FETCH_ASSOC) ?: [];
    $total_sales = (string)($rt['total_sales'] ?? '0');

    $total_float = (float)preg_replace('/[^0-9\-]/', '', $total_sales ?: '0');
    if ($total_float <= 0) {
        json_response([
            'data' => [],
            'meta' => [
                'ok' => true,
                'empty' => true,
                'reason' => 'no_sales_in_range',
                'filters' => [
                    'store_id' => $store_id,
                    'date_from' => $date_from,
                    'date_to' => $date_to,
                    'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                    'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
                ],
            ],
        ]);
    }

    // group by L1
    $sql = "
        SELECT
            ap.L1ID AS l1_id,
            MAX(ap.L1Title) AS l1_title,
            SUM(TRY_CONVERT(decimal(38,0), ap.SalesAmount)) AS sales_amount
        FROM BI.AggDailyProduct ap
        $where
        GROUP BY ap.L1ID
        ORDER BY sales_amount DESC
    ";

    // نکته: ممکنه AggDailyProduct ستون L1Title نداشته باشه
    // اگر نداشت، از DimL1 می‌گیریم.
    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        // fallback: join DimL1
        $sql2 = "
            SELECT
                ap.L1ID AS l1_id,
                MAX(dl.L1Title) AS l1_title,
                SUM(TRY_CONVERT(decimal(38,0), ap.SalesAmount)) AS sales_amount
            FROM BI.AggDailyProduct ap
            LEFT JOIN BI.DimL1 dl ON dl.L1ID = ap.L1ID
            $where
            GROUP BY ap.L1ID
            ORDER BY sales_amount DESC
        ";
        $stmt = sqlsrv_query($conn, $sql2, $params);
        if ($stmt === false) fail('DB error (l1 share)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    }

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $sales = (string)($r['sales_amount'] ?? '0');
        $sf = (float)preg_replace('/[^0-9\-]/', '', $sales ?: '0');
        $pct = ($total_float > 0) ? round(($sf / $total_float) * 100.0, 2) : 0.0;

        $items[] = [
            'l1_id' => isset($r['l1_id']) ? (int)$r['l1_id'] : null,
            'l1_title' => (string)($r['l1_title'] ?? ''),
            'sales_amount' => $sales,
            'share_pct' => $pct,
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'total_sales' => $total_sales,
            'items' => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'store_id' => $store_id,
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id'  => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'AggDailyProduct (fallback DimL1)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
