<?php
// /api/lookup/cities.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function clamp_int_allow_zero($v, $min, $max, $def) {
    // 0 = unlimited
    if ($v === null || $v === '') return $def;
    if (is_string($v) && strtolower(trim($v)) === 'all') return 0;

    $n = (int)$v;
    if ($n < 0) $n = $def;
    if ($n === 0) return 0;

    if ($n < $min) $n = $min;
    if ($n > $max) $n = $max;
    return $n;
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $conn  = db_connect();
    if (!$conn) fail('DB connection not available. (conn is null)', 500);

    // inputs
    $q = trim((string)get_param('q', ''));

    // limit: 0 means unlimited (default 0 => all)
    $limit = clamp_int_allow_zero(get_param('limit', 0), 1, 50000, 0);

    // source: dimstore (fast) | saledetail (full)
   $source = strtolower(trim((string)get_param('source', 'refcity'))); // refcity | dimstore | saledetail
if (!in_array($source, ['refcity','dimstore','saledetail'], true)) $source = 'refcity';


    // optional filters
    $province = trim((string)get_param('province', ''));
    $brand_id = get_param('brand_id');

    // role scope
    if (!empty($scope['province'])) $province = (string)$scope['province'];
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $topSql = ($limit === 0) ? "" : "TOP ($limit)";

    // -------------------------
    // Build SQL
    // -------------------------
    $params = [];
    $srcLabel = '';
if ($source === 'refcity') {

    $where = "WHERE rc.IsActive = 1 AND rc.City IS NOT NULL AND LTRIM(RTRIM(rc.City)) <> ''";
    $params = [];

    if ($province !== '') { $where .= " AND rc.Province = ?"; $params[] = $province; }
    if ($q !== '')        { $where .= " AND rc.City LIKE ?";  $params[] = '%' . $q . '%'; }

    $topSql = ($limit === 0) ? "" : "TOP ($limit)";

    $sql = "
        SELECT $topSql
          LTRIM(RTRIM(rc.City)) AS city
        FROM BI.RefCity rc WITH (NOLOCK)
        $where
        GROUP BY LTRIM(RTRIM(rc.City))
        ORDER BY LTRIM(RTRIM(rc.City)) ASC;
    ";

    $srcLabel = 'BI.RefCity (master list)';

}
    else if($source === 'saledetail') {
        // Full list from BI.SaleDetail (slow but complete)
        $where = "WHERE sd.City IS NOT NULL AND LTRIM(RTRIM(sd.City)) <> ''";

        if ($province !== '') { $where .= " AND sd.Province = ?"; $params[] = $province; }
        if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
        if ($q !== '') { $where .= " AND sd.City LIKE ?"; $params[] = '%' . $q . '%'; }

        $sql = "
            SELECT $topSql
                LTRIM(RTRIM(sd.City)) AS city
            FROM BI.SaleDetail sd WITH (NOLOCK)
            $where
            GROUP BY LTRIM(RTRIM(sd.City))
            ORDER BY LTRIM(RTRIM(sd.City)) ASC
            OPTION (RECOMPILE, MAXDOP 1);
        ";

        $srcLabel = 'BI.SaleDetail (distinct City)';
    } else {
        // Fast list from BI.DimStore
        $where = "WHERE ds.City IS NOT NULL AND LTRIM(RTRIM(ds.City)) <> ''";

        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($q !== '')        { $where .= " AND ds.City LIKE ?";  $params[] = '%' . $q . '%'; }

        // brand filter (fast) via agg
        if ($brand_id !== null && $brand_id !== '') {
            $where .= " AND EXISTS (
                SELECT 1
                FROM BI.AggDailyStoreBrandL1 a WITH (NOLOCK)
                WHERE a.StoreID = ds.StoreID AND a.BrandID = ?
            )";
            $params[] = (int)$brand_id;
        }

        $sql = "
            SELECT $topSql
                LTRIM(RTRIM(ds.City)) AS city
            FROM BI.DimStore ds WITH (NOLOCK)
            $where
            GROUP BY LTRIM(RTRIM(ds.City))
            ORDER BY LTRIM(RTRIM(ds.City)) ASC
            OPTION (RECOMPILE);
        ";

        $srcLabel = 'BI.DimStore (distinct City) + optional EXISTS on BI.AggDailyStoreBrandL1';
    }

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) fail('DB error (lookup cities)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $city = (string)($r['city'] ?? '');
        if ($city === '') continue;
        $items[] = ['value' => $city, 'label' => $city];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => ['items' => $items],
        'meta' => [
            'ok' => true,
            'filters' => [
                'q' => $q ?: null,
                'limit' => $limit,            // 0 => unlimited
                'source' => $source,          // dimstore|saledetail
                'province' => $province ?: null,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
            ],
            'duration_ms' => $elapsed_ms,
            'source' => $srcLabel,
        ]
    ]);
} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
