<?php
// /api/lookup/brands.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function clamp_int($v, $min, $max, $def) {
    $n = (int)$v;
    if ($n <= 0) $n = $def;
    if ($n < $min) $n = $min;
    if ($n > $max) $n = $max;
    return $n;
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    $q = trim((string)get_param('q', ''));
    $limit = clamp_int(get_param('limit', 50), 1, 10000, 50);

    // optional filters (برای اینکه لیست برندها با شهر/منطقه هم محدود بشه)
    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    if (!empty($scope['province'])) $province = (string)$scope['province'];

    // اگر مدیر برند => فقط همون برند
    if (!empty($scope['brand_id'])) {
        $bid = (int)$scope['brand_id'];
        json_response([
            'data' => ['items' => [['value' => $bid, 'label' => '']]], // label رو از فرانت/یا بعداً با یک query تکمیل می‌کنیم
            'meta' => ['ok' => true, 'note' => 'brand-scoped user']
        ]);
        exit;
    }

    // area فقط تهران
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    $where = "WHERE sd.BrandID IS NOT NULL";
    $params = [];

    if ($province !== '') { $where .= " AND sd.Province = ?"; $params[] = $province; }
    if ($city !== '')     { $where .= " AND sd.City = ?";     $params[] = $city; }
    if ($area !== '')     { $where .= " AND sd.Area = ?";     $params[] = $area; }
    if ($q !== '')        { $where .= " AND sd.BrandTitle LIKE ?"; $params[] = '%' . $q . '%'; }

    // برای UX بهتر: برندها رو بر اساس فروش کلی مرتب می‌کنیم
    $sql = "
        SELECT TOP ($limit)
            sd.BrandID AS brand_id,
            MAX(sd.BrandTitle) AS brand_title,
            SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS sales
        FROM BI.SaleDetail sd WITH (NOLOCK)
        $where
        GROUP BY sd.BrandID
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) fail('DB error (lookup brands)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'value' => (int)$r['brand_id'],
            'label' => (string)($r['brand_title'] ?? ''),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => ['items' => $items],
        'meta' => [
            'ok' => true,
            'filters' => [
                'q' => $q ?: null,
                'limit' => $limit,
                'province' => $province ?: null,
                'city' => $city ?: null,
                'area' => $area ?: null,
            ],
            'duration_ms' => $elapsed_ms,
            'source' => 'BI.SaleDetail (distinct BrandID/Title ordered by Sales)',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
