<?php
// /api/brands/top3m.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500, array $extra = []): void {
    $errs = sqlsrv_errors();
    fail($msg, $status, array_merge(['sqlsrv_errors' => $errs], $extra));
}

function add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // -----------------------------
    // Pagination: fixed 30 results total, 6 per page => 5 pages
    // You can override with query params but defaults match your spec.
    // -----------------------------
    $per_page = (int)get_param('per_page', 6);
    if ($per_page <= 0) $per_page = 6;
    if ($per_page > 50) $per_page = 50;

    $page = (int)get_param('page', 1);
    if ($page <= 0) $page = 1;

    $max_items = (int)get_param('max_items', 30);
    if ($max_items <= 0) $max_items = 30;
    if ($max_items > 5000) $max_items = 5000;

    $total_pages = (int)ceil($max_items / $per_page);
    if ($page > $total_pages) $page = $total_pages;

    $offset = ($page - 1) * $per_page; // rn > offset AND rn <= offset+per_page

    // -----------------------------
    // Date range: support month_from/month_to (YYYY-MM) + date_from/date_to (YYYY-MM-DD)
    // Priority: month range
    // -----------------------------
    $month_from = trim((string)get_param('month_from', ''));
    $month_to   = trim((string)get_param('month_to', ''));

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if ($month_from !== '' && $month_to !== '') {
        if (!preg_match('/^\d{4}\-\d{2}$/', $month_from) || !preg_match('/^\d{4}\-\d{2}$/', $month_to)) {
            fail('Invalid month_from/month_to. Use YYYY-MM', 400);
        }
        $date_from = $month_from . '-01';
        $date_to   = date('Y-m-t', strtotime($month_to . '-01'));
    }

    // Fallback: if not provided, use last 90 days ending at max DateKey from agg
    if (!$date_to) {
        $stmtMax = sqlsrv_query($conn, "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1");
        if ($stmtMax === false) sqlsrv_fail('DB error (max date)', 500);
        $mx = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC)['mx'] ?? null;
        if (!$mx) fail('No data to determine date range', 400);
        $date_to = (string)$mx;
    }
    if (!$date_from) {
        $date_from = add_days($date_to, -89);
    }

    $date_to_excl = add_days($date_to, 1);

    // -----------------------------
    // Filters
    // -----------------------------
    $l1_id        = get_param('l1_id');
    $brand_id     = get_param('brand_id'); // ✅ now supported
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // Optional brand title search
    $q = trim((string)get_param('q', ''));

    // Role scope
    $brand_scope = null;
    if (!empty($scope['brand_id'])) $brand_scope = (int)$scope['brand_id'];
    if ($brand_scope !== null) $brand_id = $brand_scope;

    if (!empty($scope['province'])) $province = (string)$scope['province'];

    // Area only meaningful when Tehran selected; otherwise ignore to prevent false empty results
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    // -----------------------------
    // Build WHERE + JOINS
    // -----------------------------
    $joinStore = "";
    $joinBrand = "LEFT JOIN BI.DimBrand db ON db.BrandID = a.BrandID";

    $where  = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($l1_id !== null && $l1_id !== '') {
        $where .= " AND a.L1ID = ?";
        $params[] = (int)$l1_id;
    }

    if ($sale_type_id !== null && $sale_type_id !== '') {
        $where .= " AND a.SaleTypeID = ?";
        $params[] = (int)$sale_type_id;
    }

    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
    }

    // ✅ Apply brand_id filter if provided
    if ($brand_id !== null && $brand_id !== '') {
        $where .= " AND a.BrandID = ?";
        $params[] = (int)$brand_id;
    }

    if ($q !== '') {
        $where .= " AND (db.BrandTitle LIKE ?)";
        $params[] = '%' . $q . '%';
    }

    // -----------------------------
    // SQL: totals + ranked + paged
    // Notes:
    // - totals uses the same filters (so share_percent is within the filtered scope)
    // - ranked ranks all brands within scope
    // - paged returns only up to max_items and per_page slice
    // -----------------------------
    $sql = "
        WITH totals AS (
            SELECT
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $joinBrand
            $where
        ),
        ranked AS (
            SELECT
                a.BrandID AS brand_id,
                ISNULL(NULLIF(LTRIM(RTRIM(db.BrandTitle)),''), CONCAT(N'Brand #', a.BrandID)) AS brand_title,
                SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales,
                ROW_NUMBER() OVER (ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC) AS rn
            FROM BI.AggDailyStoreBrandL1 a
            $joinStore
            $joinBrand
            $where
            GROUP BY a.BrandID, db.BrandTitle
        ),
        paged AS (
            SELECT *
            FROM ranked
            WHERE rn > ? AND rn <= ?
              AND rn <= ?
        )
        SELECT
            p.brand_id,
            p.brand_title,
            CAST(p.sales AS varchar(60)) AS sales,
            CAST(
                (CAST(p.sales AS decimal(38,6)) * 100) / NULLIF(CAST(tt.total_sales AS decimal(38,6)), 0)
            AS decimal(9,2)) AS share_percent,
            CAST(tt.total_sales AS varchar(60)) AS total_sales
        FROM paged p
        CROSS JOIN totals tt
        ORDER BY p.sales DESC
        OPTION (RECOMPILE);
    ";

    // params order:
    // - totals: params
    // - ranked: params
    // - paging bounds: offset, offset+per_page, max_items
    $paramsExec = array_merge(
        $params,
        $params,
        [$offset, $offset + $per_page, $max_items]
    );

    $stmt = sqlsrv_query($conn, $sql, $paramsExec);
    if ($stmt === false) {
        sqlsrv_fail('DB error (brands top3m)', 500, ['sql' => $sql]);
    }

    $items = [];
    $total_sales = "0";
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $total_sales = (string)($r['total_sales'] ?? "0");
        $items[] = [
            'brand_id' => (int)$r['brand_id'],
            'brand_title' => (string)$r['brand_title'],
            'sales' => (string)($r['sales'] ?? "0"),
            'share_percent' => (float)($r['share_percent'] ?? 0),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'total_sales' => $total_sales,
            'items' => $items,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $scope['province'] ?? null,
                'brand_id' => $scope['brand_id'] ?? null,
            ],
            'range' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
            ],
            'filters' => [
                'month_from' => ($month_from === '' ? null : $month_from),
                'month_to'   => ($month_to === '' ? null : $month_to),
                'date_from'  => $date_from,
                'date_to'    => $date_to,

                'l1_id' => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'brand_id' => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'q' => ($q === '' ? null : $q),

                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),

                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
            ],
            'pagination' => [
                'page' => $page,
                'per_page' => $per_page,
                'max_items' => $max_items,
                'total_pages' => $total_pages,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.AggDailyStoreBrandL1 (+DimStore/+DimBrand)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
