<?php
// /api/areas/tehran_sales.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}
function date_add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    // ---- dates ----
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if (!$date_to) {
        $stmtMax = sqlsrv_query($conn, "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1");
        if ($stmtMax === false) sqlsrv_fail('DB error (max date)', 500);
        $mx = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC)['mx'] ?? null;
        if (!$mx) fail("No data to determine date range", 400);
        $date_to = (string)$mx;
    }
    if (!$date_from) $date_from = date('Y-m-01', strtotime($date_to));
    $date_to_excl = date_add_days($date_to, 1);

    // ---- filters ----
    $l1_id = get_param('l1_id');
    $brand_id = get_param('brand_id');
    $sale_type_id = get_param('sale_type_id');

    // نقش‌ها
    if (!empty($scope['brand_id'])) $brand_id = (string)$scope['brand_id'];

    // فقط تهران
    $city = 'تهران';

    $where = "WHERE a.DateKey >= ? AND a.DateKey < ? AND ds.City = ?";
    $params = [$date_from, $date_to_excl, $city];

    if ($l1_id !== null && $l1_id !== '') { $where .= " AND a.L1ID = ?"; $params[] = (int)$l1_id; }
    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    // Area باید عددی باشه (1..22). از DimStore می‌گیریم.
    $sql = "
        SELECT
            TRY_CONVERT(int, ds.Area) AS area,
            CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS varchar(50)) AS sales,
            COUNT(DISTINCT a.StoreID) AS stores_count
        FROM BI.AggDailyStoreBrandL1 a
        JOIN BI.DimStore ds ON ds.StoreID = a.StoreID
        $where
        GROUP BY TRY_CONVERT(int, ds.Area)
        HAVING TRY_CONVERT(int, ds.Area) BETWEEN 1 AND 22
        ORDER BY area
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) sqlsrv_fail("DB error (tehran areas)", 500);

    // خروجی کامل 1..22 حتی اگر صفر بود
    $items = [];
    for ($i=1;$i<=22;$i++){
        $items[$i] = ['area'=>$i,'sales'=>"0",'stores_count'=>0];
    }

    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $a = (int)$r['area'];
        $items[$a] = [
            'area' => $a,
            'sales' => (string)($r['sales'] ?? "0"),
            'stores_count' => (int)($r['stores_count'] ?? 0),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'city' => $city,
            'items' => array_values($items),
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'range' => ['date_from'=>$date_from,'date_to'=>$date_to],
            'filters' => [
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}