<?php
// /api/ai/insight/dashboard.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/GapGPT.php';

function build_rule_based_insight(array $ctx): array {
    $filters = $ctx['filters'] ?? [];
    $kpi = $ctx['kpi'] ?? null;
    $overview = $ctx['overview'] ?? [];

    $fmt = function ($n) {
        if ($n === null || $n === '') return null;
        // numeric strings may exceed PHP int; keep as string but format if safe
        if (is_numeric($n)) {
            $s = (string)$n;
            // if very large, number_format may lose precision; do manual grouping
            if (strlen($s) > 18) {
                // simple thousands separator for integer strings
                $rev = strrev($s);
                $chunks = str_split($rev, 3);
                return strrev(implode(',', $chunks));
            }
            return number_format((float)$n, 0, '.', ',');
        }
        return (string)$n;
    };

    $cards = [];

    // headline summary
    $summaryParts = [];
    if (!empty($filters['date_from']) && !empty($filters['date_to'])) {
        $summaryParts[] = "بازه {$filters['date_from']} تا {$filters['date_to']}";
    }
    if ($kpi && isset($kpi['total_sales'])) {
        $summaryParts[] = 'فروش کل: ' . $fmt($kpi['total_sales']) . ' ریال';
    }
    if ($kpi && isset($kpi['last_week_sales'])) {
        $summaryParts[] = 'فروش هفته گذشته: ' . $fmt($kpi['last_week_sales']) . ' ریال';
    }
    if ($kpi && isset($kpi['avg_sales_per_store'])) {
        $summaryParts[] = 'میانگین هر فروشگاه: ' . $fmt($kpi['avg_sales_per_store']) . ' ریال';
    }
    $summary = implode(' | ', $summaryParts);

    // top brand
    $topBrands = $overview['top_brands_month']['items'] ?? null;
    if (is_array($topBrands) && count($topBrands)) {
        $b0 = $topBrands[0];
        $cards[] = [
            'type' => 'insight',
            'severity' => 'info',
            'title' => 'برند برتر',
            'text' => 'برند «' . ($b0['brand_title'] ?? '') . '» با فروش ' . $fmt($b0['sales'] ?? null) . ' ریال و سهم ' . ($b0['share_percent'] ?? null) . '% در صدر قرار دارد.',
            'refs' => ['top_brands_month'],
        ];
    }

    // top L1
    $l1 = $overview['l1_share']['items'] ?? null;
    if (is_array($l1) && count($l1)) {
        $c0 = $l1[0];
        $cards[] = [
            'type' => 'insight',
            'severity' => 'info',
            'title' => 'گروه کالایی برتر (L1)',
            'text' => 'گروه «' . ($c0['l1_title'] ?? '') . '» با فروش ' . $fmt($c0['sales'] ?? null) . ' ریال و سهم ' . ($c0['share_percent'] ?? null) . '% بیشترین سهم را دارد.',
            'refs' => ['l1_share'],
        ];

        $cards[] = [
            'type' => 'recommendation',
            'severity' => 'success',
            'title' => 'اقدام پیشنهادی',
            'text' => 'برای افزایش فروش، موجودی و پروموشن گروه «' . ($c0['l1_title'] ?? '') . '» را در فروشگاه‌های پرفروش تقویت کنید و اثر را هفتگی پایش کنید.',
            'refs' => ['l1_share','top_stores'],
        ];
    }

    // brand share 14d
    if (!empty($overview['brand_share_14d'])) {
        $cards[] = [
            'type' => 'insight',
            'severity' => 'info',
            'title' => 'روند سهم برندها (۱۴ روز اخیر)',
            'text' => 'روند سهم فروش برندها در ۱۴ روز اخیر آماده است و می‌توان نوسانات سهم برندهای برتر را بررسی کرد.',
            'refs' => ['brand_share_14d'],
        ];
    }

    // top store
    $topStore = $overview['top_stores']['items'] ?? null;
    if (is_array($topStore) && count($topStore)) {
        $s0 = $topStore[0];
        $cards[] = [
            'type' => 'insight',
            'severity' => 'info',
            'title' => 'فروشگاه برتر',
            'text' => 'فروشگاه «' . ($s0['store_title'] ?? '') . '» در صدر فروشگاه‌های منتخب قرار دارد. می‌توان الگوی فروش/سبد کالا را از آن استخراج کرد.',
            'refs' => ['top_stores'],
        ];
    }

    return [
        'summary' => $summary,
        'cards' => $cards,
        'kpis_used' => ['total_sales','last_week_sales','avg_sales_per_store'],
        'charts_used' => ['top_brands_month','l1_share','brand_share_14d','top_stores'],
    ];
}

function extract_json_object(string $text): ?array {
    $text = trim($text);
    if ($text === '') return null;
    // direct json
    $j = json_decode($text, true);
    if (is_array($j)) return $j;

    // try to locate first { and last }
    $start = strpos($text, '{');
    $end   = strrpos($text, '}');
    if ($start !== false && $end !== false && $end > $start) {
        $sub = substr($text, $start, $end - $start + 1);
        $j2 = json_decode($sub, true);
        if (is_array($j2)) return $j2;
    }
    return null;
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $date_from = get_param('date_from');
    $date_to   = get_param('date_to');
    if (!$date_from || !$date_to) fail('date_from and date_to are required', 400);

    // engine selection
    $engine = strtolower(trim((string)get_param('engine', 'auto')));

    $debug = (string)get_param('debug', '');
    $debug = ($debug === '1' || strtolower(trim($debug)) === 'true'); // auto|rule|gapgpt

    // build context by calling internal context endpoint directly (same server)
    $params = $_GET;
    unset($params['engine']);

    // call context endpoint via internal API function (reuse from context file pattern)
    // We'll just include and call via HTTP to avoid duplication.
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';

    $basePath = getenv('BI_BASE_PATH');
    if (!$basePath) {
        $req = $_SERVER['REQUEST_URI'] ?? '';
        $pos = strpos($req, '/api/ai/insight/dashboard.php');
        $basePath = ($pos !== false) ? substr($req, 0, $pos) : '/dashboard-project';
    }

    $qs = http_build_query(array_filter($params, fn($v) => $v !== null && $v !== ''));
    $ctxUrl = "$scheme://$host$basePath/api/ai/context/dashboard.php" . ($qs ? "?$qs" : '');

    $cookie = $_SERVER['HTTP_COOKIE'] ?? '';
    $headers = "Accept: application/json\r\n";
    if ($cookie) $headers .= "Cookie: $cookie\r\n";

    $rawCtx = @file_get_contents($ctxUrl, false, stream_context_create([
        'http' => ['method'=>'GET','header'=>$headers,'timeout'=>40]
    ]));

    $ctxArr = $rawCtx ? json_decode($rawCtx, true) : null;
    if (!is_array($ctxArr) || empty($ctxArr['meta']['ok'])) {
        fail('Failed to build context', 500, ['context_meta' => $ctxArr['meta'] ?? null, 'called_url' => $ctxUrl, 'raw' => $rawCtx]);
    }

    $ctxData = $ctxArr['data'] ?? [];

    // fallback rule-based
    $aiConf = ai_config();
    $hasKey = !empty($aiConf['api_key']);

    if ($engine === 'rule' || ($engine === 'auto' && !$hasKey)) {
        $out = build_rule_based_insight($ctxData);
        $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

        json_response([
            'data' => $out,
            'meta' => [
                'ok' => true,
                'engine' => 'ui-cards (rule-based)',
                'role' => $user['Role'] ?? null,
                'scope' => ['province'=>$scope['province'] ?? null, 'brand_id'=>$scope['brand_id'] ?? null],
                'filters' => $ctxData['filters'] ?? [],
                'generated_at' => date('Y-m-d H:i:s'),
                'duration_ms' => $elapsed_ms,
            ]
        ]);
        exit;
    }

    // GapGPT engine
    if (!$hasKey) {
        fail('GAPGPT_API_KEY is not set. Set env var or configure src/config/ai.php', 500);
    }

    $promptPath = __DIR__ . '/../../../src/ai/prompts/dashboard_insight_v1.txt';
    $prompt = @file_get_contents($promptPath);
    if ($prompt === false) $prompt = "";

    // Provide context JSON as compact string
    $contextForLLM = [
        'filters' => $ctxData['filters'] ?? null,
        'kpi' => $ctxData['kpi'] ?? null,
        'overview' => $ctxData['overview'] ?? null,
    ];

    $messages = [
        ['role' => 'system', 'content' => $prompt],
        ['role' => 'user', 'content' => "Context JSON:\n" . json_encode($contextForLLM, JSON_UNESCAPED_UNICODE)],
    ];

    $llm = gapgpt_chat($messages, [
        'temperature' => 0.2,
        'max_tokens' => 900,
    ]);

    if (!$llm['ok']) {
        // fallback to rule-based if LLM fails
        $out = build_rule_based_insight($ctxData);
        $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

        json_response([
            'data' => $out,
            'meta' => [
                'ok' => true,
                'engine' => 'ui-cards (rule-based, llm_failed)',
                'llm_error' => $llm['error'] ?? 'unknown',
                'role' => $user['Role'] ?? null,
                'scope' => ['province'=>$scope['province'] ?? null, 'brand_id'=>$scope['brand_id'] ?? null],
                'filters' => $ctxData['filters'] ?? [],
                'generated_at' => date('Y-m-d H:i:s'),
                'duration_ms' => $elapsed_ms,
            ]
        ]);
        exit;
    }

    $parsed = extract_json_object($llm['content'] ?? '');
    if (!is_array($parsed) || empty($parsed['summary'])) {
        // fallback: wrap content
        $parsed = [
            'summary' => trim((string)($llm['content'] ?? '')),
            'cards' => [],
            'kpis_used' => ['total_sales','last_week_sales','avg_sales_per_store'],
            'charts_used' => ['top_brands_month','l1_share','brand_share_14d','top_stores'],
        ];
    }

    // normalize cards
    if (!isset($parsed['cards']) || !is_array($parsed['cards'])) $parsed['cards'] = [];

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => $parsed,
        'meta' => [
            'ok' => true,
            'engine' => 'gapgpt',
            'debug' => $debug ? ['llm' => $llm] : null,
            'model' => $llm['model'] ?? ($aiConf['model'] ?? null),
            'role' => $user['Role'] ?? null,
            'scope' => ['province'=>$scope['province'] ?? null, 'brand_id'=>$scope['brand_id'] ?? null],
            'filters' => $ctxData['filters'] ?? [],
            'generated_at' => date('Y-m-d H:i:s'),
            'duration_ms' => $elapsed_ms,
            'llm_usage' => $llm['usage'] ?? null,
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
