<?php
// /api/ai/context/dashboard.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

/**
 * Call internal API endpoints (same host) and return decoded JSON + meta debug.
 */
function call_internal_api(string $path, array $params = []): array {
    // ensure no leading slash
    $path = ltrim($path, '/');

    // build URL like: http://localhost:81/dashboard-project/api/brands/top3m.php?... 
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';

    // detect project base (dashboard-project). If deployed under different folder, set BI_BASE_PATH.
    $basePath = getenv('BI_BASE_PATH');
    if (!$basePath) {
        // try infer from current script path: /dashboard-project/api/ai/context/dashboard.php
        $req = $_SERVER['REQUEST_URI'] ?? '';
        // find '/api/ai/context/dashboard.php' and keep prefix
        $pos = strpos($req, '/api/ai/context/dashboard.php');
        $basePath = ($pos !== false) ? substr($req, 0, $pos) : '/dashboard-project';
    }

    $qs = http_build_query(array_filter($params, fn($v) => $v !== null && $v !== ''));
    $url = "$scheme://$host$basePath/api/$path" . ($qs ? "?$qs" : '');

    $cookie = $_SERVER['HTTP_COOKIE'] ?? '';

    $headers = "Accept: application/json\r\n";
    if ($cookie) $headers .= "Cookie: $cookie\r\n";

    $ctx = stream_context_create([
        'http' => [
            'method'  => 'GET',
            'header'  => $headers,
            'timeout' => 35,
        ],
    ]);

    $raw = @file_get_contents($url, false, $ctx);

    // extract http code
    $code = null;
    if (isset($http_response_header) && is_array($http_response_header)) {
        foreach ($http_response_header as $line) {
            if (preg_match('/^HTTP\/\S+\s+(\d{3})/i', $line, $m)) { $code = (int)$m[1]; break; }
        }
    }

    if ($raw === false) {
        return ['data' => null, 'meta' => ['ok' => false, 'error' => 'HTTP call failed', 'http_code' => $code, 'url' => $url]];
    }

    $arr = json_decode($raw, true);
    if (!is_array($arr)) {
        return ['data' => null, 'meta' => ['ok' => false, 'error' => 'Invalid JSON from endpoint', 'http_code' => $code, 'url' => $url, 'raw' => $raw]];
    }

    // attach called_url and http_code if not present
    if (!isset($arr['meta'])) $arr['meta'] = [];
    if (is_array($arr['meta'])) {
        $arr['meta']['called_url'] = $url;
        $arr['meta']['http_code']  = $code;
    }

    return $arr;
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    // required filters
    $date_from = get_param('date_from');
    $date_to   = get_param('date_to');
    if (!$date_from || !$date_to) fail('date_from and date_to are required', 400);

    // optional filters
    $brand_id = get_param('brand_id');
    $store_id = get_param('store_id');
    $province = get_param('province');
    $city     = get_param('city');
    $area     = get_param('area');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope overrides
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];
    if (!empty($scope['province'])) $province = (string)$scope['province'];

    $baseParams = [
        'date_from' => $date_from,
        'date_to'   => $date_to,
        'brand_id'  => $brand_id,
        'store_id'  => $store_id,
        'province'  => $province,
        'city'      => $city,
        'area'      => $area,
        'l1_id'     => $l1_id,
        'sale_type_id' => $sale_type_id,
    ];

    // pack
    $kpi = call_internal_api('kpi/summary.php', $baseParams);

    $topBrands = call_internal_api('brands/top3m.php', $baseParams);     // returns top 6 by default in that endpoint (per_page=6)
    $l1Share   = call_internal_api('l1/top5.php', array_merge($baseParams, ['limit' => 5]));

    // last 14 days brand share (independent range)
    $end = new DateTime($date_to);
    $start14 = (clone $end)->modify('-13 days');
    $brand14d = call_internal_api('brands/daily_share.php', array_merge($baseParams, [
        'date_from' => $start14->format('Y-m-d'),
        'date_to'   => $end->format('Y-m-d'),
        'limit'     => 3,
    ]));

    $topStores = call_internal_api('stores/top3m.php', $baseParams);

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'user' => [
                'role' => $user['Role'] ?? null,
                'user_id' => $user['UserID'] ?? null,
                'username' => $user['Username'] ?? null,
            ],
            'scope' => [
                'province' => $scope['province'] ?? null,
                'brand_id' => $scope['brand_id'] ?? null,
            ],
            'filters' => $baseParams,

            // direct data used by AI
            'kpi' => $kpi['data'] ?? null,
            'overview' => [
                'top_brands_month' => $topBrands['data'] ?? null,
                'l1_share'         => $l1Share['data'] ?? null,
                'brand_share_14d'  => $brand14d['data'] ?? null,
                'top_stores'       => $topStores['data'] ?? null,
            ],

            // debug for troubleshooting
            'sources_meta' => [
                'kpi' => $kpi['meta'] ?? null,
                'topBrands' => $topBrands['meta'] ?? null,
                'l1Share' => $l1Share['meta'] ?? null,
                'brand14d' => $brand14d['meta'] ?? null,
                'topStores' => $topStores['meta'] ?? null,
            ],
        ],
        'meta' => [
            'ok' => true,
            'generated_at' => date('Y-m-d H:i:s'),
            'duration_ms' => $elapsed_ms,
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
