<?php
// /api/stores/selected/summary.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/config/db.php';
require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/DateRange.php';


try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $store_id = get_param('store_id');
    if ($store_id === null || $store_id === '') {
        fail('store_id is required', 400);
    }
    $store_id = (int)$store_id;

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    // default: last 90 days
    if (!$date_from || !$date_to) {
        $date_to = date('Y-m-d');
        $date_from = date('Y-m-d', strtotime('-89 days'));
    }
    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));

    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // role scope (brand manager)
    if (!empty($scope['brand_id'])) $brand_id = (int)$scope['brand_id'];

    $conn = db_connect();

    // ---------------------------
    // 1) total sales from AggDailyStoreBrandL1 (fast)
    // ---------------------------
    $where = "WHERE a.StoreID = ? AND a.DateKey >= ? AND a.DateKey < ?";
    $params = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND a.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND a.L1ID = ?";    $params[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where .= " AND a.SaleTypeID = ?"; $params[] = (int)$sale_type_id; }

    $sqlTotal = "
        SELECT
            SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS total_sales,
            COUNT(DISTINCT a.DateKey) AS days_count
        FROM BI.AggDailyStoreBrandL1 a
        $where
    ";

    $st = sqlsrv_query($conn, $sqlTotal, $params);
    if ($st === false) fail('DB error (store summary total)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rt = sqlsrv_fetch_array($st, SQLSRV_FETCH_ASSOC) ?: [];

    $total_sales = (string)($rt['total_sales'] ?? '0');
    $days_count  = (int)($rt['days_count'] ?? 0);

    // avg daily sales (safe)
    $avg_daily_sales = '0';
    if ($days_count > 0) {
        // avoid php int overflow -> float safe enough for avg
        $ts = preg_replace('/[^0-9\-]/', '', $total_sales);
        if ($ts === '' || $ts === '-') $ts = '0';
        $avg_daily_sales = (string)intval(((float)$ts) / $days_count);
    }

    // ---------------------------
    // 2) last 7 days vs prev 7 days (relative to date_to)
    // ---------------------------
    $last7_from = date('Y-m-d', strtotime($date_to . ' -6 days'));
    $prev7_from = date('Y-m-d', strtotime($date_to . ' -13 days'));
    $prev7_to_excl = date('Y-m-d', strtotime($date_to . ' -6 days')); // exclusive

    // last 7d
    $where7 = "WHERE a.StoreID = ? AND a.DateKey >= ? AND a.DateKey < ?";
    $p7 = [$store_id, $last7_from, $date_to_excl];
    if ($brand_id !== null && $brand_id !== '') { $where7 .= " AND a.BrandID = ?"; $p7[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where7 .= " AND a.L1ID = ?";    $p7[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $where7 .= " AND a.SaleTypeID = ?"; $p7[] = (int)$sale_type_id; }

    $sql7 = "SELECT SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales_7d
             FROM BI.AggDailyStoreBrandL1 a
             $where7";

    $st7 = sqlsrv_query($conn, $sql7, $p7);
    if ($st7 === false) fail('DB error (store summary last7d)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $r7 = sqlsrv_fetch_array($st7, SQLSRV_FETCH_ASSOC) ?: [];
    $sales_last_7d = (string)($r7['sales_7d'] ?? '0');

    // prev 7d
    $whereP = "WHERE a.StoreID = ? AND a.DateKey >= ? AND a.DateKey < ?";
    $pp = [$store_id, $prev7_from, $prev7_to_excl];
    if ($brand_id !== null && $brand_id !== '') { $whereP .= " AND a.BrandID = ?"; $pp[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $whereP .= " AND a.L1ID = ?";    $pp[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $whereP .= " AND a.SaleTypeID = ?"; $pp[] = (int)$sale_type_id; }

    $sqlP = "SELECT SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales_prev_7d
             FROM BI.AggDailyStoreBrandL1 a
             $whereP";

    $stP = sqlsrv_query($conn, $sqlP, $pp);
    if ($stP === false) fail('DB error (store summary prev7d)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rP = sqlsrv_fetch_array($stP, SQLSRV_FETCH_ASSOC) ?: [];
    $sales_prev_7d = (string)($rP['sales_prev_7d'] ?? '0');

    // growth %
    $growth_pct = 0.0;
    $a7 = (float)preg_replace('/[^0-9\-]/', '', $sales_last_7d ?: '0');
    $p7f = (float)preg_replace('/[^0-9\-]/', '', $sales_prev_7d ?: '0');
    if ($p7f > 0) $growth_pct = (($a7 - $p7f) / $p7f) * 100.0;

    $avg_sales_per_day_last_7d = '0';
    $avg_sales_per_day_last_7d = (string)intval($a7 / 7.0);

    // ---------------------------
    // 3) SKU count (AggDailyProduct) - still fast because StoreID+DateKey is selective
    // ---------------------------
    $whereSku = "WHERE ap.StoreID = ? AND ap.DateKey >= ? AND ap.DateKey < ?";
    $ps = [$store_id, $date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $whereSku .= " AND ap.BrandID = ?"; $ps[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $whereSku .= " AND ap.L1ID = ?";    $ps[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $whereSku .= " AND ap.SaleTypeID = ?"; $ps[] = (int)$sale_type_id; }

    $sqlSku = "
        SELECT APPROX_COUNT_DISTINCT(ap.ProductID) AS sku_count
        FROM BI.AggDailyProduct ap
        $whereSku
    ";

    $stS = sqlsrv_query($conn, $sqlSku, $ps);
    if ($stS === false) fail('DB error (store summary sku)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $rs = sqlsrv_fetch_array($stS, SQLSRV_FETCH_ASSOC) ?: [];
    $sku_count = (int)($rs['sku_count'] ?? 0);

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'store_id' => $store_id,
            'total_sales' => $total_sales,
            'total_sales_3m' => $total_sales,
            'days_count' => $days_count,
            'avg_daily_sales' => $avg_daily_sales,
            'sales_last_7d' => $sales_last_7d,
            'sales_prev_7d' => $sales_prev_7d,
            'growth_pct_vs_prev_week' => round($growth_pct, 2),
            'avg_sales_per_day_last_7d' => $avg_sales_per_day_last_7d,
            'sku_count' => $sku_count,
            'date_window' => [
                'last7_from' => $last7_from,
                'last7_to'   => $date_to,
                'prev7_from' => $prev7_from,
                'prev7_to'   => date('Y-m-d', strtotime($prev7_to_excl . ' -1 day')),
            ],
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'],
            'scope' => [
                'province' => $scope['province'],
                'brand_id' => $scope['brand_id'],
            ],
            'filters' => [
                'store_id' => $store_id,
                'date_from' => $date_from,
                'date_to'   => $date_to,
                'brand_id'  => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'     => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'AggDailyStoreBrandL1 + AggDailyProduct',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
