<?php
// /api/sales/timeseries.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function sqlsrv_fail(string $msg, int $status = 500): void {
    fail($msg, $status, ['sqlsrv_errors' => sqlsrv_errors()]);
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    // اگر پارامتر نداد: دیفالت = این ماه تا امروز در برابر ماه قبل (کل ماه قبل)
    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if (!$date_from || !$date_to) {
        $today = date('Y-m-d');

        $date_from = date('Y-m-01'); // اول همین ماه
        $date_to   = $today;         // تا امروز

        $prev_from = date('Y-m-01', strtotime('first day of last month')); // اول ماه قبل
        $prev_to   = date('Y-m-t',  strtotime('last day of last month'));  // آخر ماه قبل
    } else {
        // اگر بازه سفارشی داد: بازه قبلی هم‌طول بساز
        $from_ts = strtotime($date_from);
        $to_ts   = strtotime($date_to);
        $days = (int)floor(($to_ts - $from_ts) / 86400) + 1;
        if ($days < 1) $days = 1;

        $prev_to   = date('Y-m-d', strtotime($date_from . ' -1 day'));
        $prev_from = date('Y-m-d', strtotime($date_from . " -{$days} days"));
    }

    $date_to_excl = date('Y-m-d', strtotime($date_to . ' +1 day'));
    $prev_to_excl = date('Y-m-d', strtotime($prev_to . ' +1 day'));

    // فیلترها (صفحه ۱)
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id');
    $city     = trim((string)get_param('city', ''));
    $province = trim((string)get_param('province', ''));
    $area     = trim((string)get_param('area', '')); // فقط تهران
    $sale_type_id = get_param('sale_type_id');

    // role scope
    if ($scope['province']) $province = $scope['province'];
    if ($scope['brand_id']) $brand_id = $scope['brand_id'];

    $conn = db_connect();

    $joinStore = "";
    $whereExtra = "";
    $paramsExtra = [];

    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $whereExtra .= " AND ds.Province = ?"; $paramsExtra[] = $province; }
        if ($city !== '')     { $whereExtra .= " AND ds.City = ?";     $paramsExtra[] = $city; }
        if ($area !== '')     { $whereExtra .= " AND ds.Area = ?";     $paramsExtra[] = $area; }
    }

    if ($brand_id !== null && $brand_id !== '') { $whereExtra .= " AND a.BrandID = ?"; $paramsExtra[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $whereExtra .= " AND a.L1ID = ?";    $paramsExtra[] = (int)$l1_id; }
    if ($sale_type_id !== null && $sale_type_id !== '') { $whereExtra .= " AND a.SaleTypeID = ?"; $paramsExtra[] = (int)$sale_type_id; }

    $sql = "
        SELECT
            a.DateKey AS [date],
            CAST(SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS varchar(50)) AS sales
        FROM BI.AggDailyStoreBrandL1 a
        $joinStore
        WHERE a.DateKey >= ? AND a.DateKey < ?
        $whereExtra
        GROUP BY a.DateKey
        ORDER BY a.DateKey
    ";

    $run = function(string $from, string $to_excl) use ($conn, $sql, $paramsExtra) {
        $params = array_merge([$from, $to_excl], $paramsExtra);
        $stmt = sqlsrv_query($conn, $sql, $params);
        if ($stmt === false) sqlsrv_fail('DB error (timeseries)', 500);

        $out = [];
        while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
            $out[] = [
                'date'  => $row['date'],
                'sales' => $row['sales'] ?? '0',
            ];
        }
        return $out;
    };

    $current  = $run($date_from, $date_to_excl);
    $previous = $run($prev_from, $prev_to_excl);

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'current' => $current,
            'previous' => $previous,
        ],
        'meta' => [
            'ok' => true,
            'role' => $scope['role'],
            'ranges' => [
                'current' => ['from' => $date_from, 'to' => $date_to],
                'previous' => ['from' => $prev_from, 'to' => $prev_to],
            ],
            'filters' => [
                'brand_id' => ($brand_id === '' ? null : $brand_id),
                'l1_id' => ($l1_id === '' ? null : $l1_id),
                'province' => ($province === '' ? null : $province),
                'city' => ($city === '' ? null : $city),
                'area' => ($area === '' ? null : $area),
                'sale_type_id' => ($sale_type_id === '' ? null : $sale_type_id),
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
