<?php
// /api/products/performance_monthly.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

function resolve_range_or_fail(): array {
    $month_from = trim((string)get_param('month_from', ''));
    $month_to   = trim((string)get_param('month_to', ''));

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if ($month_from !== '' && $month_to !== '') {
        if (!preg_match('/^\d{4}\-\d{2}$/', $month_from) || !preg_match('/^\d{4}\-\d{2}$/', $month_to)) {
            fail('Invalid month_from/month_to. Use YYYY-MM', 400);
        }
        $date_from = $month_from . '-01';
        $date_to   = date('Y-m-t', strtotime($month_to . '-01'));
    }

    if (!$date_from || !$date_to) {
        fail('month_from/month_to (or date_from/date_to) are required', 400);
    }

    $date_to_excl = add_days($date_to, 1);
    return [$date_from, $date_to, $date_to_excl];
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    [$date_from, $date_to, $date_to_excl] = resolve_range_or_fail();

    // filters
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id'); // optional
    $l2_id    = get_param('l2_id'); // optional

    $city = trim((string)get_param('city', ''));
    $area = trim((string)get_param('area', ''));

    // scope
    $province_scope = !empty($scope['province']) ? (string)$scope['province'] : '';
    $brand_scope    = !empty($scope['brand_id']) ? (int)$scope['brand_id'] : null;
    if ($brand_scope !== null) $brand_id = $brand_scope;

    // area فقط تهران
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    $topN = (int)get_param('top', 6);
    if ($topN <= 0) $topN = 6;
    if ($topN > 50) $topN = 50;

    // WHERE base
    $where  = "WHERE sd.FactorDate >= ? AND sd.FactorDate < ?";
    $params = [$date_from, $date_to_excl];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND sd.ProductGroupID = ?"; $params[] = (int)$l1_id; }
    if ($l2_id !== null && $l2_id !== '')       { $where .= " AND sd.ProductCategoryID = ?"; $params[] = (int)$l2_id; }

    if ($province_scope !== '') { $where .= " AND sd.Province = ?"; $params[] = $province_scope; }
    if ($city !== '')           { $where .= " AND sd.City = ?";     $params[] = $city; }
    if ($area !== '')           { $where .= " AND sd.Area = ?";     $params[] = $area; }

    // SQL:
    // 1) top_products by total sales in range (and also total qty)
    // 2) monthly aggregation for those products (sales + qty)
    // 3) prev month using LAG(sales/qty) over months per product
    $sql = "
        WITH top_products AS (
            SELECT TOP ($topN)
                sd.ProductID AS product_id,
                MAX(sd.ProductTitle) AS product_title,
                MAX(sd.BrandID) AS brand_id,
                MAX(sd.BrandTitle) AS brand_title,
                MAX(sd.ProductGroupID) AS l1_id,
                MAX(sd.ProductGroupTitle) AS l1_title,
                MAX(sd.ProductCategoryID) AS l2_id,
                MAX(sd.ProductCategoryTitle) AS l2_title,
                SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS total_sales,
                SUM(TRY_CONVERT(decimal(38,6), sd.Quantity)) AS total_qty
            FROM BI.SaleDetail sd WITH (NOLOCK)
            $where
              AND sd.ProductID IS NOT NULL
            GROUP BY sd.ProductID
            ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
        ),
        monthly_raw AS (
            SELECT
                CONVERT(char(7), sd.FactorDate, 120) AS ym,   -- YYYY-MM
                sd.ProductID AS product_id,
                SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS sales,
                SUM(TRY_CONVERT(decimal(38,6), sd.Quantity)) AS qty
            FROM BI.SaleDetail sd WITH (NOLOCK)
            JOIN top_products tp ON tp.product_id = sd.ProductID
            $where
            GROUP BY CONVERT(char(7), sd.FactorDate, 120), sd.ProductID
        ),
        monthly AS (
            SELECT
                ym,
                product_id,
                sales,
                qty,
                LAG(sales) OVER (PARTITION BY product_id ORDER BY ym) AS prev_sales,
                LAG(qty)   OVER (PARTITION BY product_id ORDER BY ym) AS prev_qty
            FROM monthly_raw
        )
        SELECT
            tp.product_id,
            tp.product_title,
            tp.brand_id,
            tp.brand_title,
            tp.l1_id,
            tp.l1_title,
            tp.l2_id,
            tp.l2_title,
            CAST(tp.total_sales AS varchar(60)) AS total_sales_range,
            CAST(tp.total_qty   AS varchar(60)) AS total_qty_range,

            m.ym,
            CAST(m.sales AS varchar(60)) AS sales,
            CAST(m.qty   AS varchar(60)) AS qty,
            CAST(ISNULL(m.prev_sales, 0) AS varchar(60)) AS prev_sales,
            CAST(ISNULL(m.prev_qty,   0) AS varchar(60)) AS prev_qty,
            CAST((m.sales - ISNULL(m.prev_sales,0)) AS varchar(60)) AS delta_sales,
            CAST(
                CASE
                    WHEN ISNULL(m.prev_sales,0) = 0 THEN NULL
                    ELSE (CAST((m.sales - m.prev_sales) AS decimal(38,6)) * 100) / NULLIF(CAST(m.prev_sales AS decimal(38,6)), 0)
                END
            AS decimal(9,2)) AS growth_percent
        FROM top_products tp
        LEFT JOIN monthly m ON m.product_id = tp.product_id
        ORDER BY tp.total_sales DESC, m.ym ASC
        OPTION (RECOMPILE);
    ";

    // where twice => params twice
    $paramsExec = array_merge($params, $params);

    $stmt = sqlsrv_query($conn, $sql, $paramsExec);
    if ($stmt === false) {
        fail('DB error (product performance monthly)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);
    }

    // Build output
    $products = [];

    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $pid = (int)($r['product_id'] ?? 0);
        if ($pid <= 0) continue;

        if (!isset($products[$pid])) {
            $products[$pid] = [
                'product_id' => $pid,
                'product_title' => (string)($r['product_title'] ?? ''),
                'brand_id' => (int)($r['brand_id'] ?? 0),
                'brand_title' => (string)($r['brand_title'] ?? ''),
                'l1_id' => (int)($r['l1_id'] ?? 0),
                'l1_title' => (string)($r['l1_title'] ?? ''),
                'l2_id' => ($r['l2_id'] === null ? null : (int)$r['l2_id']),
                'l2_title' => (string)($r['l2_title'] ?? ''),
                'total_sales_range' => (string)($r['total_sales_range'] ?? '0'),
                'total_qty_range' => (string)($r['total_qty_range'] ?? '0'),
                'monthly' => [],
            ];
        }

        $ym = (string)($r['ym'] ?? '');
        if ($ym !== '') {
            $products[$pid]['monthly'][] = [
                'ym' => $ym,
                'sales' => (string)($r['sales'] ?? '0'),
                'qty' => (string)($r['qty'] ?? '0'),
                'prev_sales' => (string)($r['prev_sales'] ?? '0'),
                'prev_qty' => (string)($r['prev_qty'] ?? '0'),
                'delta_sales' => (string)($r['delta_sales'] ?? '0'),
                'growth_percent' => ($r['growth_percent'] === null ? null : (float)$r['growth_percent']),
            ];
        }
    }

    $products_list = array_values($products);

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'top' => $topN,
            'products' => $products_list,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $province_scope ?: null,
                'brand_id' => $brand_scope,
            ],
            'filters' => [
                'month_from' => get_param('month_from') ?: null,
                'month_to'   => get_param('month_to') ?: null,
                'date_from'  => $date_from,
                'date_to'    => $date_to,
                'brand_id'   => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'      => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'l2_id'      => ($l2_id !== null && $l2_id !== '') ? (int)$l2_id : null,
                'city'       => $city ?: null,
                'area'       => $area ?: null,
                'top'        => $topN,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.SaleDetail',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
